InstallMethod( \=,
   "for two nilpotent orbits",
   true, [ IsNilpotentOrbit, IsNilpotentOrbit ], 0,
   function( a, b )

   return AmbientLieAlgebra(a) = AmbientLieAlgebra(b) and
          WeightedDynkinDiagram(a) = WeightedDynkinDiagram(b);

end );

InstallMethod( NilpotentOrbit,
   "for a Lie algebra and list",
   true, [ IsLieAlgebra, IsList ], 0,
   function( L, list )

      local o, fam;

      if not IsBound( L!.nilorbType ) then
         fam:= NewFamily( "nilorbsfam", IsNilpotentOrbit );
         fam!.invCM:= CartanMatrix( RootSystem(L) )^-1;
         L!.nilorbType:= NewType( fam, 
                           IsNilpotentOrbit and IsAttributeStoringRep );
      fi;
      o:= Objectify( L!.nilorbType, rec() );
      SetAmbientLieAlgebra( o, L );
      SetWeightedDynkinDiagram( o, list );
      if HasSemiSimpleType(L) then
         SetSemiSimpleType( o, SemiSimpleType(L) );
      fi; 
      return o;

end );


InstallMethod( SL2Triple,
   "for a Lie algebra and a nilpotent element",
   true, [ IsLieAlgebra, IsObject ], 0,

function( L, x )

  # Same as FindSl2 in the library; only returns the elements, rather
  # than the subalgebra. (Advantage: GAP doesn't change the basis.)

   local n,         # the dimension of `L'
         F,         # the field of `L'
         B,         # basis of `L'
         T,         # the table of structure constants of `L'
         xc,        # coefficient vector
         eqs,       # a system of equations
         i,j,k,l,   # loop variables
         cij,       # the element `T[i][j]'
         b,         # the right hand side of the equation system
         v,         # solution of the equations
         z,         # element of `L'
         h,         # element of `L'
         R,         # centralizer of `x' in `L'
         BR,        # basis of `R'
         Rvecs,     # basis vectors of `R'
         H,         # the matrix of `ad H' restricted to `R'
         e0,        # coefficient vector
         e1,        # coefficient vector
         y;         # element of `L'

    if not IsNilpotentElement( L, x ) then
      Error( "<x> must be a nilpotent element of the Lie algebra <L>" );
    fi;

    n:= Dimension( L );
    F:= LeftActingDomain( L );
    B:= Basis( L );
    T:= StructureConstantsTable( B );

    xc:= Coefficients( B, x );
    eqs:= NullMat( 2*n, 2*n, F );

    # First we try to find elements `z' and `h' such that `[x,z]=h'
    # and `[h,x]=2x' (i.e., such that two of the three defining equations
    # of sl_2 are satisfied).
    # This results in a system of `2n' equations for `2n' variables.

    for i in [1..n] do
      for j in [1..n] do
        cij:= T[i][j];
        for k in [1..Length(cij[1])] do
          l:= cij[1][k];
          eqs[i][l] := eqs[i][l] + xc[j]*cij[2][k];
          eqs[n+i][n+l]:= eqs[n+i][n+l] + xc[j]*cij[2][k];
        od;
      od;
      eqs[n+i][i]:= One( F );
    od;

    b:= [];
    for i in [1..n] do
      b[i]:= Zero( F );
      b[n+i]:= 2*One( F )*xc[i];
    od;

    v:= SolutionMat( eqs, b );

    if v = fail then
      # There is no sl_2 containing <x>.
      return fail;
    fi;

    z:= LinearCombination( B, v{ [   1 ..   n ] } );
    h:= LinearCombination( B, v{ [ n+1 .. 2*n ] } );

    R:= LieCentralizer( L, SubalgebraNC( L, [ x ] ) );
    BR:= Basis( R );
    Rvecs:= BasisVectors( BR );

    # `ad h' maps `R' into `R'. `H' will be the matrix of that map.

    H:= List( Rvecs, v -> Coefficients( BR, h * v ) );

    # By the proof of the lemma of Jacobson-Morozov (see Jacobson,
    # Lie Algebras, p. 98) there is an element `e1' in `R' such that
    # `(H+2)e1=e0' where `e0=[h,z]+2z'.
    # If we set `y=z-e1' then `x,h,y' will span a subalgebra of `L'
    # isomorphic to sl_2.

    H:= H+2*IdentityMat( Dimension( R ), F );


    e0:= Coefficients( BR, h * z + 2*z );
    e1:= SolutionMat( H, e0 );

    if e1 = fail then
      # There is no sl_2 containing <x>.
      return fail;
    fi;

    y:= z-LinearCombination(Rvecs,e1);

    return [y,h,x];

end );



InstallMethod( PrintObj,
        "for nilpotent orbit",
        true,
       [ IsNilpotentOrbit ], 0,
       function( o )


     Print("<nilpotent orbit in Lie algebra");
     if HasSemiSimpleType(o) then
        Print( " of type ", SemiSimpleType(o) );
     fi;
     Print(">");

end );

InstallMethod( RandomSL2Triple,
   "for a nilpotent orbit",
   true, [ IsNilpotentOrbit], 0,
   function( o )

   local Ci, L, c, H, v, h, adh, sp, i, j, k, e, f, found, found_good, co,
         x, eqns, sol, allcft1;

   Ci:= FamilyObj( o )!.invCM;
   L:= AmbientLieAlgebra( o );
   c:= ChevalleyBasis(L);
   H:= c[3];

   v:= Ci*WeightedDynkinDiagram( o );
   h:= Sum( List( [1..Length(H)], x -> v[x]*H[x] ) );
   adh:= AdjointMatrix( Basis(L), h );
   sp:= NullspaceMat( TransposedMat( adh-2*adh^0 ) );
   e:= List( sp, x -> Sum( List( [1..Dimension(L)], ii -> x[ii]*Basis(L)[ii] ) ) );
   sp:= NullspaceMat( TransposedMat( adh+2*adh^0 ) );
   f:= List( sp, x -> Sum( List( [1..Dimension(L)], ii -> x[ii]*Basis(L)[ii] ) ) );

   allcft1:= false;
   
   while not allcft1 do
      found:= false;
      j:= 1;
      while not found do 
            co:= List( [1..Length(e)], x-> Random( Rationals ) );
            x:= Sum( List( [1..Length(e)], ii -> co[ii]*e[ii] ) );
            sp:= Subspace( L, List( f, y -> x*y ) );
            if Dimension(sp) = Length(e) and h in sp then
               found:= true; 
            else
               j:= j+1;
               if j > 100 then
                  Error("Tried hard but found no representative of the nilpotent orbit. Are you sure the weighted Dynkin diagram really corresponds to a nilpotent orbit?");
               fi;
            fi;
      od;

      for k in [Length(co),Length(co)-1..1] do
          i:= -1; found_good:= false;
          while not found_good do
                i:= i+1;
                co[k]:= i;
                x:= Sum( List( [1..Length(e)], ii -> co[ii]*e[ii] ) );
                sp:= Subspace( L, List( f, y -> x*y ) );
                if Dimension(sp) = Length(e) and h in sp then
                   found_good:= true; 
                fi;   
          od;
      od;
      allcft1:= ForAll( co, x -> (x=0) or (x=1) );

   od;
 
   eqns:= List( f, u -> Coefficients( Basis(sp), x*u ) );
   sol:= SolutionMat( eqns, Coefficients( Basis(sp), h ) ); 
   return [Sum([1..Length(f)], ii -> sol[ii]*f[ii] ),h,x]; 


end );

InstallMethod( SL2Triple,
   "for a nilpotent orbit",
   true, [ IsNilpotentOrbit], 0,
   function( o )


   return RandomSL2Triple( o );

end );



SLAfcts.weighted_dyn_diags_Dn:= function( n )


    local part, good, p, q, k, new, cur, m, diags, is_very_even, d, 
          d0, h, a, b, diag;

    part:= Partitions( 2*n );
    new:= [ ];
    for p in part do
        q:= [ ];
        cur:= p[1];
        k:= 1;
        m:= 0;
        while k <= Length(p) do
            m:= m+1;
            if k < Length(p) and p[k+1] <> cur then
               Add( q, [ cur, m ] );
               cur:= p[k+1];
               m:= 0;
            fi;
            k:= k+1;
        od;
        Add( q, [ cur,m ] );
        good:= true;
        for k in [1..Length(q)] do           
            if IsEvenInt(q[k][1]) and not IsEvenInt(q[k][2]) then
               good:= false;
               break;
            fi;
        od;
        if good then
           Add( new, p );
        fi;
    od;
    part:= new;

    diags:= [ ];
    new:= [ ];
    for p in part do
        is_very_even:= ForAll( p, x -> IsEvenInt(x) );
        d:= [ ];
        for k in p do
            d0:= List( [1..k], i -> k -2*(i-1)-1 );
            Append( d, d0 );
        od;
        Sort(d);
        d:= Reversed( d );
        h:= d{[1..n]};
        if not is_very_even then
           diag:= List( [1..n-1], i -> h[i]-h[i+1] );
           Add( diag, h[n-1]+h[n] );
           Add( diags, diag );
           Add( new, p );
        else
           if IsInt(n/4) then a:= 0; else a:= 2; fi;
           b:= 2-a;
           diag:= List( [1..n-2], i -> h[i]-h[i+1] );
           Add( diag, a ); Add( diag, b ); 
           Add( diags, diag );
           Add( new, p );
           diag:= List( [1..n-2], i -> h[i]-h[i+1] );
           Add( diag, b ); Add( diag, a ); 
           Add( diags, diag );
           Add( new, p );
        fi;
    od;

    return [diags,new];


end;


SLAfcts.weighted_dyn_diags_An:= function( n )


    local part, diags, p, d, k, d0, h;

    part:= Partitions( n+1 );
    diags:= [ ];
    for p in part do
        d:= [ ];
        for k in p do
            d0:= List( [1..k], i -> k -2*(i-1)-1 );
            Append( d, d0 );
        od;
        Sort(d);
        h:= Reversed( d );
        Add( diags, List( [1..n], i -> h[i]-h[i+1] ) );
    od;

    return [diags,part];


end;

SLAfcts.weighted_dyn_diags_Bn:= function( n )


    local part, good, p, q, k, new, cur, m, diags, is_very_even, d, 
          d0, h, a, b, diag;

    part:= Partitions( 2*n+1 );
    new:= [ ];
    for p in part do
        q:= [ ];
        cur:= p[1];
        k:= 1;
        m:= 0;
        while k <= Length(p) do
            m:= m+1;
            if k < Length(p) and p[k+1] <> cur then
               Add( q, [ cur, m ] );
               cur:= p[k+1];
               m:= 0;
            fi;
            k:= k+1;
        od;
        Add( q, [ cur,m ] );
        good:= true;
        for k in [1..Length(q)] do           
            if IsEvenInt(q[k][1]) and not IsEvenInt(q[k][2]) then
               good:= false;
               break;
            fi;
        od;
        if good then
           Add( new, p );
        fi;
    od;
    part:= new;

    diags:= [ ];
    for p in part do
        d:= [ ];
        for k in p do
            d0:= List( [1..k], i -> k -2*(i-1)-1 );
            Append( d, d0 );
        od;
        # Get rid of a zero;
        for k in [1..Length(d)] do
            if d[k] = 0 then
               RemoveElmList( d, k );
               break;
            fi;
        od;
        Sort(d);
        d:= Reversed( d );
        h:= d{[1..n]};
        diag:= List( [1..n-1], i -> h[i]-h[i+1] );
        Add( diag, h[n] );
        Add( diags, diag );
    od;

    return [diags,part];


end;


SLAfcts.weighted_dyn_diags_Cn:= function( n )


    local part, good, p, q, k, new, cur, m, diags, is_very_even, d, 
          d0, h, a, b, diag;

    part:= Partitions( 2*n );
    new:= [ ];
    for p in part do
        q:= [ ];
        cur:= p[1];
        k:= 1;
        m:= 0;
        while k <= Length(p) do
            m:= m+1;
            if k < Length(p) and p[k+1] <> cur then
               Add( q, [ cur, m ] );
               cur:= p[k+1];
               m:= 0;
            fi;
            k:= k+1;
        od;
        Add( q, [ cur,m ] );
        good:= true;
        for k in [1..Length(q)] do           
            if IsOddInt(q[k][1]) and not IsEvenInt(q[k][2]) then
               good:= false;
               break;
            fi;
        od;
        if good then
           Add( new, p );
        fi;
    od;
    part:= new;

    diags:= [ ];
    for p in part do
        d:= [ ];
        for k in p do
            d0:= List( [1..k], i -> k -2*(i-1)-1 );
            Append( d, d0 );
        od;
        Sort(d);
        d:= Reversed( d );
        h:= d{[1..n]};
        diag:= List( [1..n-1], i -> h[i]-h[i+1] );
        Add( diag, 2*h[n] );
        Add( diags, diag );
    od;

    return [diags,part];

end;



SLAfcts.nilporbs:=  function( L )

      
        # This for simple types only (at least for the moment).
        # REMARK: nonzero nilpotent orbits!
        # This is only for the standard simple Lie alg!

      local type, list, elms, orbs, i, j, o, u, a, sl2;

      type:= SemiSimpleType( L );
      if type = "G2" then

         list:= [ [1,0], [0,1], [0,2], [2,2] ]; 
         elms:=
[ [ [ 10, 1 ], [ 13, 2, 14, 3 ], [ 4, 1 ] ],
  [ [ 12, 1 ], [ 13, 1, 14, 2 ], [ 6, 1 ] ], 
  [ [ 8, 1, 10, 1 ], [ 13, 2, 14, 4 ], [ 2, 1, 4, 1 ] ], 
  [ [ 7, 6, 8, 10 ], [ 13, 6, 14, 10 ], [ 1, 1, 2, 1 ] ] ];

      elif type = "F4" then

         list:= [ [1,0,0,0], [0,0,0,1], [0,1,0,0], [2,0,0,0], [0,0,0,2],
                  [0,0,1,0], [2,0,0,1], [0,1,0,1], [1,0,1,0], [0,2,0,0], [2,2,0,0],
                  [1,0,1,2], [0,2,0,2], [2,2,0,2], [2,2,2,2] ];
         # need to permute, to account for slightly strange numbering of the Dynkin
         # diagram in GAP:
         list:= List( list, x -> Permuted( x, (1,2,4) ) );
         elms:=
[ [ [ 48, 1 ], [ 49, 1, 50, 2, 51, 2, 52, 3 ], [ 24, 1 ] ], 
  [ [ 45, 1 ], [ 49, 2, 50, 2, 51, 3, 52, 4 ], [ 21, 1 ] ], 
  [ [ 41, 1, 46, 1 ], [ 49, 2, 50, 3, 51, 4, 52, 6 ], [ 17, 1, 22, 1 ] ], 
  [ [ 40, 2, 42, 2 ], [ 49, 2, 50, 4, 51, 4, 52, 6 ], [ 16, 1, 18, 1 ] ], 
  [ [ 35, 2, 36, 2 ], [ 49, 4, 50, 4, 51, 6, 52, 8 ], [ 11, 1, 12, 1 ] ], 
  [ [ 38, 1, 39, 2, 40, 2 ], [ 49, 3, 50, 4, 51, 6, 52, 8 ], 
      [ 14, 1, 15, 1, 16, 1 ] ], 
  [ [ 33, 3, 39, 4 ], [ 49, 4, 50, 6, 51, 7, 52, 10 ], [ 9, 1, 15, 1 ] ], 
  [ [ 32, 2, 38, 2, 40, 1 ], [ 49, 4, 50, 5, 51, 7, 52, 10 ], 
      [ 8, 1, 14, 1, 16, 1 ] ], 
  [ [ 34, 4, 35, 3, 39, 1 ], [ 49, 4, 50, 6, 51, 8, 52, 11 ], 
      [ 10, 1, 11, 1, 15, 1 ] ], 
  [ [ 32, 2, 33, 2, 34, 2, 42, 2 ], [ 49, 4, 50, 6, 51, 8, 52, 12 ], 
      [ 8, 1, 9, 1, 10, 1, 18, 1 ] ], 
  [ [ 26, 10, 32, 6, 34, 6 ], [ 49, 6, 50, 10, 51, 12, 52, 18 ], 
      [ 2, 1, 8, 1, 10, 1 ] ], 
  [ [ 25, 8, 33, 5, 34, 9 ], [ 49, 8, 50, 10, 51, 14, 52, 19 ], 
      [ 1, 1, 9, 1, 10, 1 ] ], 
  [ [ 29, 8, 30, 9, 31, 5, 37, 1 ], [ 49, 8, 50, 10, 51, 14, 52, 20 ], 
      [ 5, 1, 6, 1, 7, 1, 13, 1 ] ], 
  [ [ 26, 14, 28, 18, 29, 10, 34, 8 ], [ 49, 10, 50, 14, 51, 18, 52, 26 ], 
      [ 2, 1, 4, 1, 5, 1, 10, 1 ] ], 
  [ [ 25, 16, 26, 22, 27, 30, 28, 42 ], [ 49, 16, 50, 22, 51, 30, 52, 42 ], 
      [ 1, 1, 2, 1, 3, 1, 4, 1 ] ] ];
  
      elif type = "E6" then

         list:= [ [0,1,0,0,0,0], [1,0,0,0,0,1], [0,0,0,1,0,0], 
                  [0,2,0,0,0,0], [1,1,0,0,0,1], [2,0,0,0,0,2], [0,0,1,0,1,0],
                  [1,2,0,0,0,1], [1,0,0,1,0,1], [0,1,1,0,1,0], [0,0,0,2,0,0], 
                  [2,2,0,0,0,2], [0,2,0,2,0,0], [1,1,1,0,1,1], [2,1,1,0,1,2], 
                  [1,2,1,0,1,1], [2,0,0,2,0,2], [2,2,0,2,0,2], [2,2,2,0,2,2], 
                  [2,2,2,2,2,2] ];
         elms:=
[ [ [ 72, 1 ], [ 73, 1, 74, 2, 75, 2, 76, 3, 77, 2, 78, 1 ], [ 36, 1 ] ], 
  [ [ 66, 1, 70, 1 ], [ 73, 2, 74, 2, 75, 3, 76, 4, 77, 3, 78, 2 ], 
      [ 30, 1, 34, 1 ] ], 
  [ [ 60, 1, 68, 1, 69, 1 ], [ 73, 2, 74, 3, 75, 4, 76, 6, 77, 4, 78, 2 ], 
      [ 24, 1, 32, 1, 33, 1 ] ], 
  [ [ 61, 2, 62, 2 ], [ 73, 2, 74, 4, 75, 4, 76, 6, 77, 4, 78, 2 ], 
      [ 25, 1, 26, 1 ] ], 
  [ [ 59, 1, 61, 2, 62, 2 ], [ 73, 3, 74, 4, 75, 5, 76, 7, 77, 5, 78, 3 ], 
      [ 23, 1, 25, 1, 26, 1 ] ], 
  [ [ 53, 2, 54, 2, 56, 2, 57, 2 ], [ 73, 4, 74, 4, 75, 6, 76, 8, 77, 6, 78, 
          4 ], [ 17, 1, 18, 1, 20, 1, 21, 1 ] ], 
  [ [ 58, 1, 59, 2, 60, 2, 61, 1 ], 
      [ 73, 3, 74, 4, 75, 6, 76, 8, 77, 6, 78, 3 ], 
      [ 22, 1, 23, 1, 24, 1, 25, 1 ] ], 
  [ [ 49, 3, 50, 3, 59, 4 ], [ 73, 4, 74, 6, 75, 7, 76, 10, 77, 7, 78, 4 ], 
      [ 13, 1, 14, 1, 23, 1 ] ], 
  [ [ 48, 2, 56, 2, 57, 2, 58, 2, 60, 1 ], [ 73, 4, 74, 5, 75, 7, 76, 10, 77, 
          7, 78, 4 ], [ 12, 1, 20, 1, 21, 1, 22, 1, 24, 1 ] ], 
  [ [ 51, 4, 53, 3, 56, 3, 59, 1 ], 
      [ 73, 4, 74, 6, 75, 8, 76, 11, 77, 8, 78, 4 ], 
      [ 15, 1, 17, 1, 20, 1, 23, 1 ] ], 
  [ [ 45, 1, 52, 4, 53, 3, 54, 1, 55, 3 ], 
      [ 73, 4, 74, 6, 75, 8, 76, 12, 77, 8, 78, 4 ], 
      [ 9, 1, 16, 1, 17, 1, 18, 1, 19, 1 ] ], 
  [ [ 47, 6, 48, 6, 49, 4, 50, 4 ], 
      [ 73, 6, 74, 8, 75, 10, 76, 14, 77, 10, 78, 6 ], 
      [ 11, 1, 12, 1, 13, 1, 14, 1 ] ], 
  [ [ 38, 10, 48, 6, 51, 6, 52, 6 ], [ 73, 6, 74, 10, 75, 12, 76, 18, 77, 12, 
          78, 6 ], [ 2, 1, 12, 1, 15, 1, 16, 1 ] ], 
  [ [ 47, 6, 48, 6, 49, 4, 50, 4, 51, 1 ], 
      [ 73, 6, 74, 8, 75, 11, 76, 15, 77, 11, 78, 6 ], 
      [ 11, 1, 12, 1, 13, 1, 14, 1, 15, 1 ] ], 
  [ [ 37, 8, 42, 8, 49, 5, 50, 5, 51, 9 ], 
      [ 73, 8, 74, 10, 75, 14, 76, 19, 77, 14, 78, 8 ], 
      [ 1, 1, 6, 1, 13, 1, 14, 1, 15, 1 ] ], 
  [ [ 38, -5, 43, 5, 44, 10, 47, 7, 48, 2, 51, 6, 52, 2 ], 
      [ 73, 7, 74, 10, 75, 13, 76, 18, 77, 13, 78, 7 ], 
      [ 7, 1, 8, 1, 11, 1, 12, 1, 15, 1 ] ], 
  [ [ 37, 8, 44, 5, 45, 9, 46, 1, 47, 8, 55, 5 ], 
      [ 73, 8, 74, 10, 75, 14, 76, 20, 77, 14, 78, 8 ], 
      [ 1, 1, 8, 1, 9, 1, 10, 1, 11, 1, 19, 1 ] ], 
  [ [ 38, 14, 42, 10, 43, 10, 45, 8, 46, 18 ], 
      [ 73, 10, 74, 14, 75, 18, 76, 26, 77, 18, 78, 10 ], 
      [ 2, 1, 6, 1, 7, 1, 9, 1, 10, 1 ] ], 
  [ [ 37, 12, 38, 8, 39, -8, 41, 22, 42, 12, 44, 8, 45, 22, 46, 8 ], 
      [ 73, 12, 74, 16, 75, 22, 76, 30, 77, 22, 78, 12 ], 
      [ 1, 1, 2, 1, 5, 1, 6, 1, 8, 1, 9, 1 ] ], 
  [ [ 37, 16, 38, 22, 39, 30, 40, 42, 41, 30, 42, 16 ], 
      [ 73, 16, 74, 22, 75, 30, 76, 42, 77, 30, 78, 16 ], 
      [ 1, 1, 2, 1, 3, 1, 4, 1, 5, 1, 6, 1 ] ] ];

      elif type = "E7" then

         list:= [ [1,0,0,0,0,0,0], [0,0,0,0,0,1,0], [0,0,0,0,0,0,2], [0,0,1,0,0,0,0],
                  [2,0,0,0,0,0,0], [0,1,0,0,0,0,1], [1,0,0,0,0,1,0], [0,0,0,1,0,0,0], 
                  [2,0,0,0,0,1,0], [0,0,0,0,0,2,0], [0,2,0,0,0,0,0], [2,0,0,0,0,0,2],
                  [0,0,1,0,0,1,0], [1,0,0,1,0,0,0], [0,0,2,0,0,0,0], [1,0,0,0,1,0,1],
                  [2,0,2,0,0,0,0], [0,1,1,0,0,0,1], [0,0,0,1,0,1,0], [2,0,0,0,0,2,0],
                  [0,0,0,0,2,0,0], [2,0,0,0,0,2,2], [2,1,1,0,0,0,1], [1,0,0,1,0,1,0],
                  [2,0,0,1,0,1,0], [0,0,0,2,0,0,0], [1,0,0,1,0,2,0], [1,0,0,1,0,1,2],
                  [2,0,0,0,2,0,0], [0,1,1,0,1,0,2], [0,0,2,0,0,2,0], [2,0,2,0,0,2,0], 
                  [0,0,0,2,0,0,2], [0,0,0,2,0,2,0], [2,1,1,0,1,1,0], [2,1,1,0,1,0,2],
                  [2,0,0,2,0,0,2], [2,1,1,0,1,2,2], [2,0,0,2,0,2,0], [2,0,2,2,0,2,0],
                  [2,0,0,2,0,2,2], [2,2,2,0,2,0,2], [2,2,2,0,2,2,2], [2,2,2,2,2,2,2] ];

      elms:=
[ [ [ 126, 1 ], [ 127, 2, 128, 2, 129, 3, 130, 4, 131, 3, 132, 2, 133, 1 ], 
      [ 63, 1 ] ], 
  [ [ 120, 1, 123, 1 ], [ 127, 2, 128, 3, 129, 4, 130, 6, 131, 5, 132, 4, 
          133, 2 ], [ 57, 1, 60, 1 ] ], 
  [ [ 110, 1, 111, 1, 112, 1 ], [ 127, 2, 128, 3, 129, 4, 130, 6, 131, 5, 
          132, 4, 133, 3 ], [ 47, 1, 48, 1, 49, 1 ] ], 
  [ [ 105, 1, 119, 1, 122, 1 ], 
      [ 127, 3, 128, 4, 129, 6, 130, 8, 131, 6, 132, 4, 133, 2 ], 
      [ 42, 1, 56, 1, 59, 1 ] ], 
  [ [ 107, 2, 109, 2 ], [ 127, 4, 128, 4, 129, 6, 130, 8, 131, 6, 132, 4, 
          133, 2 ], [ 44, 1, 46, 1 ] ], 
  [ [ 108, 1, 110, 1, 115, 1, 116, 1 ], [ 127, 3, 128, 5, 129, 6, 130, 9, 
          131, 7, 132, 5, 133, 3 ], [ 45, 1, 47, 1, 52, 1, 53, 1 ] ], 
  [ [ 107, 2, 109, 2, 112, 1 ], 
      [ 127, 4, 128, 5, 129, 7, 130, 10, 131, 8, 132, 6, 133, 3 ], 
      [ 44, 1, 46, 1, 49, 1 ] ], 
  [ [ 104, 2, 105, 1, 106, 2, 114, 1 ], [ 127, 4, 128, 6, 129, 8, 130, 12, 
          131, 9, 132, 6, 133, 3 ], [ 41, 1, 42, 1, 43, 1, 51, 1 ] ], 
  [ [ 83, 3, 84, 3, 112, 4 ], 
      [ 127, 6, 128, 7, 129, 10, 130, 14, 131, 11, 132, 8, 133, 4 ], 
      [ 20, 1, 21, 1, 49, 1 ] ], 
  [ [ 97, 2, 98, 2, 99, 2, 106, 2 ], [ 127, 4, 128, 6, 129, 8, 130, 12, 131, 
          10, 132, 8, 133, 4 ], [ 34, 1, 35, 1, 36, 1, 43, 1 ] ], 
  [ [ 100, 1, 101, 1, 102, 2, 103, 2, 104, 1 ], 
      [ 127, 4, 128, 7, 129, 8, 130, 12, 131, 9, 132, 6, 133, 3 ], 
      [ 37, 1, 38, 1, 39, 1, 40, 1, 41, 1 ] ], 
  [ [ 90, 3, 93, 4, 94, 1, 100, 3 ], 
      [ 127, 6, 128, 7, 129, 10, 130, 14, 131, 11, 132, 8, 133, 5 ], 
      [ 27, 1, 30, 1, 31, 1, 37, 1 ] ], 
  [ [ 97, 2, 98, 2, 99, 2, 100, 1, 106, 2 ], 
      [ 127, 5, 128, 7, 129, 10, 130, 14, 131, 11, 132, 8, 133, 4 ], 
      [ 34, 1, 35, 1, 36, 1, 37, 1, 43, 1 ] ], 
  [ [ 90, 3, 91, 4, 102, 3, 112, 1 ], 
      [ 127, 6, 128, 8, 129, 11, 130, 16, 131, 12, 132, 8, 133, 4 ], 
      [ 27, 1, 28, 1, 39, 1, 49, 1 ] ], 
  [ [ 90, 1, 92, 3, 94, 3, 95, 4, 102, 1 ], 
      [ 127, 6, 128, 8, 129, 12, 130, 16, 131, 12, 132, 8, 133, 4 ], 
      [ 27, 1, 29, 1, 31, 1, 32, 1, 39, 1 ] ], 
  [ [ 90, 3, 93, 4, 94, 1, 100, 3, 103, 1 ], 
      [ 127, 6, 128, 8, 129, 11, 130, 16, 131, 13, 132, 9, 133, 5 ], 
      [ 27, 1, 30, 1, 31, 1, 37, 1, 40, 1 ] ], 
  [ [ 64, 10, 91, 6, 92, 6, 94, 6 ], 
      [ 127, 10, 128, 12, 129, 18, 130, 24, 131, 18, 132, 12, 133, 6 ], 
      [ 1, 1, 28, 1, 29, 1, 31, 1 ] ], 
  [ [ 85, 1, 93, 1, 94, 4, 95, 3, 96, 3, 98, 1 ], 
      [ 127, 6, 128, 9, 129, 12, 130, 17, 131, 13, 132, 9, 133, 5 ], 
      [ 22, 1, 30, 1, 31, 1, 32, 1, 33, 1, 35, 1 ] ], 
  [ [ 90, 2, 92, 3, 93, 2, 94, 3, 95, 4 ], 
      [ 127, 6, 128, 9, 129, 12, 130, 18, 131, 14, 132, 10, 133, 5 ], 
      [ 27, 1, 29, 1, 30, 1, 31, 1, 32, 1 ] ], 
  [ [ 83, 4, 84, 4, 88, 6, 92, 6 ], 
      [ 127, 8, 128, 10, 129, 14, 130, 20, 131, 16, 132, 12, 133, 6 ], 
      [ 20, 1, 21, 1, 25, 1, 29, 1 ] ], 
  [ [ 88, 4, 89, 3, 90, 2, 91, 2, 92, 3, 110, 1 ], 
      [ 127, 6, 128, 9, 129, 12, 130, 18, 131, 15, 132, 10, 133, 5 ], 
      [ 25, 1, 26, 1, 27, 1, 28, 1, 29, 1, 47, 1 ] ], 
  [ [ 70, 9, 83, 5, 84, 5, 86, 8, 87, 8 ], 
      [ 127, 10, 128, 13, 129, 18, 130, 26, 131, 21, 132, 16, 133, 9 ], 
      [ 7, 1, 20, 1, 21, 1, 23, 1, 24, 1 ] ], 
  [ [ 64, 10, 91, 6, 92, 6, 93, 1, 94, 6 ], 
      [ 127, 10, 128, 13, 129, 18, 130, 25, 131, 19, 132, 13, 133, 7 ], 
      [ 1, 1, 28, 1, 29, 1, 30, 1, 31, 1 ] ], 
  [ [ 83, 4, 84, 4, 88, 6, 91, 1, 92, 6 ], 
      [ 127, 8, 128, 11, 129, 15, 130, 22, 131, 17, 132, 12, 133, 6 ], 
      [ 20, 1, 21, 1, 25, 1, 28, 1, 29, 1 ] ], 
  [ [ 71, 10, 81, 6, 91, 6, 92, 1, 93, 6, 94, 1 ], 
      [ 127, 10, 128, 13, 129, 18, 130, 26, 131, 20, 132, 14, 133, 7 ], 
      [ 8, 1, 18, 1, 28, 1, 29, 1, 30, 1, 31, 1 ] ], 
  [ [ 83, 6, 84, 2, 85, 4, 86, 2, 87, 4, 88, 6 ], 
      [ 127, 8, 128, 12, 129, 16, 130, 24, 131, 18, 132, 12, 133, 6 ], 
      [ 20, 1, 21, 1, 22, 1, 23, 1, 24, 1, 25, 1 ] ], 
  [ [ 75, 8, 76, 8, 83, 5, 84, 5, 91, 9 ], 
      [ 127, 10, 128, 14, 129, 19, 130, 28, 131, 22, 132, 16, 133, 8 ], 
      [ 12, 1, 13, 1, 20, 1, 21, 1, 28, 1 ] ], 
  [ [ 70, 9, 81, 8, 83, 5, 84, 5, 91, 1, 92, 8 ], 
      [ 127, 10, 128, 14, 129, 19, 130, 28, 131, 22, 132, 16, 133, 9 ], 
      [ 7, 1, 18, 1, 20, 1, 21, 1, 28, 1, 29, 1 ] ], 
  [ [ 71, 5, 80, 7, 81, 6, 82, 5, 83, 10, 91, -2, 92, 1, 93, 2 ], 
      [ 127, 10, 128, 13, 129, 18, 130, 26, 131, 21, 132, 14, 133, 7 ], 
      [ 17, 1, 18, 1, 19, 1, 20, 1, 29, 1, 30, 1 ] ], 
  [ [ 70, 6, 76, 9, 78, 6, 79, 6, 83, 3, 84, 10, 86, 3, 87, 4, 90, -3 ], 
      [ 127, 10, 128, 15, 129, 20, 130, 29, 131, 23, 132, 16, 133, 9 ], 
      [ 13, 1, 15, 1, 16, 1, 21, 1, 23, 1, 24, 1 ] ], 
  [ [ 77, 9, 78, 5, 80, 5, 82, 8, 86, 8, 89, 1 ], 
      [ 127, 10, 128, 14, 129, 20, 130, 28, 131, 22, 132, 16, 133, 8 ], 
      [ 14, 1, 15, 1, 17, 1, 19, 1, 23, 1, 26, 1 ] ], 
  [ [ 64, 14, 78, 18, 80, 8, 81, 10, 82, 10 ], 
      [ 127, 14, 128, 18, 129, 26, 130, 36, 131, 28, 132, 20, 133, 10 ], 
      [ 1, 1, 15, 1, 17, 1, 18, 1, 19, 1 ] ], 
  [ [ 76, 6, 77, 6, 78, 10, 79, 4, 81, 6, 82, 9, 84, 3, 85, -3, 87, 3, 96, 1 ]
        , [ 127, 10, 128, 15, 129, 20, 130, 30, 131, 23, 132, 16, 133, 9 ], 
      [ 14, 1, 15, 1, 16, 1, 18, 1, 19, 1, 21, 1, 33, 1 ] ], 
  [ [ 75, 10, 76, 10, 77, 12, 78, 6, 79, 12, 80, 6 ], 
      [ 127, 12, 128, 18, 129, 24, 130, 36, 131, 28, 132, 20, 133, 10 ], 
      [ 12, 1, 13, 1, 14, 1, 15, 1, 16, 1, 17, 1 ] ], 
  [ [ 64, 14, 78, 18, 79, 1, 80, 8, 81, 10, 82, 10 ], 
      [ 127, 14, 128, 19, 129, 26, 130, 37, 131, 29, 132, 20, 133, 10 ], 
      [ 1, 1, 15, 1, 16, 1, 17, 1, 18, 1, 19, 1 ] ], 
  [ [ 64, 14, 70, 2, 76, 9, 78, 8, 79, 2, 80, 18, 86, 11, 87, 9 ], 
      [ 127, 14, 128, 19, 129, 26, 130, 37, 131, 29, 132, 20, 133, 11 ], 
      [ 1, 1, 7, 1, 13, 1, 15, 1, 17, 1, 23, 1 ] ], 
  [ [ 67, -9, 71, 14, 74, 18, 76, 11, 78, 9, 82, 2, 85, 2, 86, 8, 87, 1 ], 
      [ 127, 14, 128, 19, 129, 26, 130, 38, 131, 29, 132, 20, 133, 11 ], 
      [ 8, 1, 11, 1, 13, 1, 15, 1, 22, 1, 23, 1, 24, 1 ] ], 
  [ [ 64, 18, 69, 28, 70, 15, 78, 10, 79, 15, 80, 24 ], 
      [ 127, 18, 128, 25, 129, 34, 130, 49, 131, 39, 132, 28, 133, 15 ], 
      [ 1, 1, 6, 1, 7, 1, 15, 1, 16, 1, 17, 1 ] ], 
  [ [ 67, 15, 71, 16, 75, 12, 76, 12, 78, 7, 79, 15, 80, 7 ], 
      [ 127, 16, 128, 22, 129, 30, 130, 44, 131, 34, 132, 24, 133, 12 ], 
      [ 4, 1, 8, 1, 12, 1, 13, 1, 15, 1, 16, 1, 17, 1 ] ], 
  [ [ 64, 22, 66, 42, 72, 30, 74, 30, 75, 16, 76, 16 ], 
      [ 127, 22, 128, 30, 129, 42, 130, 60, 131, 46, 132, 32, 133, 16 ], 
      [ 1, 1, 3, 1, 9, 1, 11, 1, 12, 1, 13, 1 ] ], 
  [ [ 70, 15, 71, 18, 72, 24, 73, 15, 74, 10, 75, 28, 85, 1 ], 
      [ 127, 18, 128, 25, 129, 34, 130, 50, 131, 39, 132, 28, 133, 15 ], 
      [ 7, 1, 8, 1, 9, 1, 10, 1, 11, 1, 12, 1, 22, 1 ] ], 
  [ [ 64, 22, 65, 28, 66, -28, 70, 17, 72, 3, 73, 42, 74, 15, 75, 32, 76, 3, 
          81, 3 ], 
      [ 127, 22, 128, 31, 129, 42, 130, 60, 131, 47, 132, 32, 133, 17 ], 
      [ 1, 1, 2, 1, 7, 1, 9, 1, 10, 1, 11, 1, 12, 1 ] ], 
  [ [ 64, 26, 65, 15, 66, -15, 68, 57, 69, 40, 70, 21, 72, 22, 73, 50, 74, 22 
         ], [ 127, 26, 128, 37, 129, 50, 130, 72, 131, 57, 132, 40, 133, 21 ],
      [ 1, 1, 2, 1, 5, 1, 6, 1, 7, 1, 9, 1, 10, 1 ] ], 
  [ [ 64, 34, 65, 49, 66, 66, 67, 96, 68, 75, 69, 52, 70, 27 ], 
      [ 127, 34, 128, 49, 129, 66, 130, 96, 131, 75, 132, 52, 133, 27 ], 
      [ 1, 1, 2, 1, 3, 1, 4, 1, 5, 1, 6, 1, 7, 1 ] ] ];

      elif type = "E8" then

         list:= [ [0,0,0,0,0,0,0,1], [1,0,0,0,0,0,0,0], [0,0,0,0,0,0,1,0], [0,0,0,0,0,0,0,2],
                  [0,1,0,0,0,0,0,0], [1,0,0,0,0,0,0,1], [0,0,0,0,0,1,0,0], [1,0,0,0,0,0,0,2],
                  [0,0,1,0,0,0,0,0], [2,0,0,0,0,0,0,0], [1,0,0,0,0,0,1,0], [0,0,0,0,0,1,0,1],
                  [0,0,0,0,0,0,2,0], [0,0,0,0,0,0,2,2], [0,0,0,0,1,0,0,0], [0,0,1,0,0,0,0,1],
                  [0,1,0,0,0,0,1,0], [1,0,0,0,0,1,0,0], [2,0,0,0,0,0,0,2], [0,0,0,1,0,0,0,0], 
                  [0,1,0,0,0,0,1,2], [0,2,0,0,0,0,0,0], [1,0,0,0,0,1,0,1], [1,0,0,0,1,0,0,0],
                  [1,0,0,0,0,1,0,2], [0,0,0,1,0,0,0,1], [0,0,0,0,0,2,0,0], [2,0,0,0,0,1,0,1],
                  [0,0,0,1,0,0,0,2], [0,0,1,0,0,1,0,0], [0,2,0,0,0,0,0,2], [2,0,0,0,0,0,2,0],
                  [2,0,0,0,0,0,2,2], [0,0,0,1,0,0,1,0], [1,0,0,1,0,0,0,1], [0,0,1,0,0,1,0,1],
                  [0,1,1,0,0,0,1,0], [1,0,0,0,1,0,1,0], [0,0,0,1,0,1,0,0], [1,0,0,0,1,0,1,2],
                  [0,0,0,0,2,0,0,0], [2,0,0,0,0,2,0,0], [0,1,1,0,0,0,1,2], [1,0,0,1,0,1,0,0],
                  [0,0,0,1,0,1,0,2], [2,0,0,0,0,2,0,2], [0,0,0,0,2,0,0,2], [2,1,1,0,0,0,1,2],
                  [2,0,0,0,0,2,2,2], [1,0,0,1,0,1,0,1], [1,0,0,1,0,1,1,0], [1,0,0,1,0,1,0,2],
                  [2,0,0,1,0,1,0,2], [0,0,0,2,0,0,0,2], [2,0,0,0,2,0,0,2], [1,0,0,1,0,1,2,2], 
                  [0,1,1,0,1,0,2,2], [0,0,0,2,0,0,2,0], [2,1,1,0,1,1,0,1], [0,0,0,2,0,0,2,2],
                  [2,1,1,0,1,0,2,2], [2,0,0,2,0,0,2,0], [2,0,0,2,0,0,2,2], [2,1,1,0,1,2,2,2],
                  [2,0,0,2,0,2,0,2], [2,0,0,2,0,2,2,2], [2,2,2,0,2,0,2,2], [2,2,2,0,2,2,2,2],
                  [2,2,2,2,2,2,2,2] ];

         elms:= 
[ [ [ 240, 1 ], 
      [ 241, 2, 242, 3, 243, 4, 244, 6, 245, 5, 246, 4, 247, 3, 248, 2 ], 
      [ 120, 1 ] ], 
  [ [ 233, 1, 234, 1 ], [ 241, 4, 242, 5, 243, 7, 244, 10, 245, 8, 246, 6, 
          247, 4, 248, 2 ], [ 113, 1, 114, 1 ] ], 
  [ [ 224, 1, 225, 1, 226, 1 ], 
      [ 241, 4, 242, 6, 243, 8, 244, 12, 245, 10, 246, 8, 247, 6, 248, 3 ], 
      [ 104, 1, 105, 1, 106, 1 ] ], 
  [ [ 208, 2, 210, 2 ], [ 241, 4, 242, 6, 243, 8, 244, 12, 245, 10, 246, 8, 
          247, 6, 248, 4 ], [ 88, 1, 90, 1 ] ], 
  [ [ 215, 1, 217, 1, 218, 1, 223, 1 ], 
      [ 241, 5, 242, 8, 243, 10, 244, 15, 245, 12, 246, 9, 247, 6, 248, 3 ], 
      [ 95, 1, 97, 1, 98, 1, 103, 1 ] ], 
  [ [ 208, 2, 210, 2, 217, 1 ], [ 241, 6, 242, 8, 243, 11, 244, 16, 245, 13, 
          246, 10, 247, 7, 248, 4 ], [ 88, 1, 90, 1, 97, 1 ] ], 
  [ [ 203, 2, 205, 2, 211, 1, 212, 1 ], 
      [ 241, 6, 242, 9, 243, 12, 244, 18, 245, 15, 246, 12, 247, 8, 248, 4 ], 
      [ 83, 1, 85, 1, 91, 1, 92, 1 ] ], 
  [ [ 162, 3, 163, 3, 217, 4 ], [ 241, 8, 242, 11, 243, 15, 244, 22, 245, 18, 
          246, 14, 247, 10, 248, 6 ], [ 42, 1, 43, 1, 97, 1 ] ], 
  [ [ 197, 2, 200, 2, 202, 1, 204, 1, 210, 1 ], 
      [ 241, 7, 242, 10, 243, 14, 244, 20, 245, 16, 246, 12, 247, 8, 248, 4 ],
      [ 77, 1, 80, 1, 82, 1, 84, 1, 90, 1 ] ], 
  [ [ 190, 2, 191, 2, 193, 2, 201, 2 ], 
      [ 241, 8, 242, 10, 243, 14, 244, 20, 245, 16, 246, 12, 247, 8, 248, 4 ],
      [ 70, 1, 71, 1, 73, 1, 81, 1 ] ], 
  [ [ 190, 2, 192, 2, 193, 2, 194, 1, 200, 2 ], 
      [ 241, 8, 242, 11, 243, 15, 244, 22, 245, 18, 246, 14, 247, 10, 248, 5 ]
        , [ 70, 1, 72, 1, 73, 1, 74, 1, 80, 1 ] ], 
  [ [ 182, 3, 185, 3, 186, 4, 213, 1 ], 
      [ 241, 8, 242, 12, 243, 16, 244, 24, 245, 20, 246, 16, 247, 11, 248, 6 ]
        , [ 62, 1, 65, 1, 66, 1, 93, 1 ] ], 
  [ [ 180, 4, 181, 3, 182, 1, 184, 3, 198, 1 ], 
      [ 241, 8, 242, 12, 243, 16, 244, 24, 245, 20, 246, 16, 247, 12, 248, 6 ]
        , [ 60, 1, 61, 1, 62, 1, 64, 1, 78, 1 ] ], 
  [ [ 128, 10, 173, 6, 175, 6, 191, 6 ], 
      [ 241, 12, 242, 18, 243, 24, 244, 36, 245, 30, 246, 24, 247, 18, 248, 
          10 ], [ 8, 1, 53, 1, 55, 1, 71, 1 ] ], 
  [ [ 189, 2, 190, 2, 191, 1, 192, 1, 193, 2, 194, 2 ], 
      [ 241, 8, 242, 12, 243, 16, 244, 24, 245, 20, 246, 15, 247, 10, 248, 5 ]
        , [ 69, 1, 70, 1, 71, 1, 72, 1, 73, 1, 74, 1 ] ], 
  [ [ 178, 4, 187, 3, 188, 3, 189, 1, 202, 1 ], 
      [ 241, 9, 242, 13, 243, 18, 244, 26, 245, 21, 246, 16, 247, 11, 248, 6 ]
        , [ 58, 1, 67, 1, 68, 1, 69, 1, 82, 1 ] ], 
  [ [ 176, 4, 179, 3, 189, 1, 191, 3, 192, 1, 193, 1 ], 
      [ 241, 9, 242, 14, 243, 18, 244, 27, 245, 22, 246, 17, 247, 12, 248, 6 ]
        , [ 56, 1, 59, 1, 69, 1, 71, 1, 72, 1, 73, 1 ] ], 
  [ [ 165, 3, 177, 3, 178, 2, 187, 2, 194, 4 ], 
      [ 241, 10, 242, 14, 243, 19, 244, 28, 245, 23, 246, 18, 247, 12, 248, 6 
         ], [ 45, 1, 57, 1, 58, 1, 67, 1, 74, 1 ] ], 
  [ [ 162, 4, 163, 4, 173, 6, 177, 6 ], 
      [ 241, 12, 242, 16, 243, 22, 244, 32, 245, 26, 246, 20, 247, 14, 248, 8 
         ], [ 42, 1, 43, 1, 53, 1, 57, 1 ] ], 
  [ [ 172, 3, 178, 4, 180, 2, 181, 2, 182, 3, 198, 1 ], 
      [ 241, 10, 242, 15, 243, 20, 244, 30, 245, 24, 246, 18, 247, 12, 248, 6 
         ], [ 52, 1, 58, 1, 60, 1, 61, 1, 62, 1, 78, 1 ] ], 
  [ [ 128, 10, 175, 6, 178, 6, 186, 6, 189, 1 ], 
      [ 241, 13, 242, 20, 243, 26, 244, 39, 245, 32, 246, 25, 247, 18, 248, 
          10 ], [ 8, 1, 55, 1, 58, 1, 66, 1, 69, 1 ] ], 
  [ [ 165, 1, 174, 2, 176, 4, 177, 3, 178, 1, 179, 3, 181, 2 ], 
      [ 241, 10, 242, 16, 243, 20, 244, 30, 245, 24, 246, 18, 247, 12, 248, 6 
         ], [ 45, 1, 54, 1, 56, 1, 57, 1, 58, 1, 59, 1, 61, 1 ] ], 
  [ [ 162, 4, 163, 4, 173, 6, 177, 6, 181, 1 ], 
      [ 241, 12, 242, 17, 243, 23, 244, 34, 245, 28, 246, 22, 247, 15, 248, 8 
         ], [ 42, 1, 43, 1, 53, 1, 57, 1, 61, 1 ] ], 
  [ [ 164, 3, 165, 3, 166, 3, 167, 3, 175, 4, 188, 4 ], 
      [ 241, 12, 242, 17, 243, 23, 244, 34, 245, 28, 246, 21, 247, 14, 248, 7 
         ], [ 44, 1, 45, 1, 46, 1, 47, 1, 55, 1, 68, 1 ] ], 
  [ [ 135, 10, 165, 6, 177, 6, 178, 1, 179, 1, 181, 6 ], 
      [ 241, 14, 242, 20, 243, 27, 244, 40, 245, 33, 246, 26, 247, 18, 248, 
          10 ], [ 15, 1, 45, 1, 57, 1, 58, 1, 59, 1, 61, 1 ] ], 
  [ [ 167, 4, 169, 6, 170, 4, 171, 1, 172, 6, 184, 1 ], 
      [ 241, 12, 242, 18, 243, 24, 244, 36, 245, 29, 246, 22, 247, 15, 248, 8 
         ], [ 47, 1, 49, 1, 50, 1, 51, 1, 52, 1, 64, 1 ] ], 
  [ [ 165, 2, 166, 4, 167, 6, 168, 6, 169, 4, 170, 2 ], 
      [ 241, 12, 242, 18, 243, 24, 244, 36, 245, 30, 246, 24, 247, 16, 248, 8 
         ], [ 45, 1, 46, 1, 47, 1, 48, 1, 49, 1, 50, 1 ] ], 
  [ [ 143, 8, 144, 8, 162, 5, 163, 5, 181, 9 ], 
      [ 241, 16, 242, 22, 243, 30, 244, 44, 245, 36, 246, 28, 247, 19, 248, 
          10 ], [ 23, 1, 24, 1, 42, 1, 43, 1, 61, 1 ] ], 
  [ [ 142, 10, 149, 5, 160, -2, 164, 6, 168, 7, 173, 2, 179, 5, 191, 1 ], 
      [ 241, 14, 242, 21, 243, 28, 244, 42, 245, 34, 246, 26, 247, 18, 248, 
          10 ], [ 22, 1, 44, 1, 48, 1, 53, 1, 59, 1, 71, 1 ] ], 
  [ [ 164, 1, 165, 2, 166, 4, 167, 6, 168, 6, 169, 4, 170, 2 ], 
      [ 241, 13, 242, 19, 243, 26, 244, 38, 245, 31, 246, 24, 247, 16, 248, 8 
         ], [ 44, 1, 45, 1, 46, 1, 47, 1, 48, 1, 49, 1, 50, 1 ] ], 
  [ [ 154, 6, 157, 6, 158, 6, 160, 2, 163, 10, 184, 2 ], 
      [ 241, 14, 242, 22, 243, 28, 244, 42, 245, 34, 246, 26, 247, 18, 248, 
          10 ], [ 34, 1, 37, 1, 38, 1, 40, 1, 43, 1, 64, 1 ] ], 
  [ [ 154, 5, 155, 5, 156, 9, 158, 8, 164, 8, 170, 1 ], 
      [ 241, 16, 242, 22, 243, 30, 244, 44, 245, 36, 246, 28, 247, 20, 248, 
          10 ], [ 34, 1, 35, 1, 36, 1, 38, 1, 44, 1, 50, 1 ] ], 
  [ [ 128, 14, 141, 18, 157, 10, 169, 8, 171, 10 ], 
      [ 241, 20, 242, 28, 243, 38, 244, 56, 245, 46, 246, 36, 247, 26, 248, 
          14 ], [ 8, 1, 21, 1, 37, 1, 49, 1, 51, 1 ] ], 
  [ [ 159, 6, 161, 3, 162, 6, 163, 3, 164, 4, 165, 4, 168, 4 ], 
      [ 241, 14, 242, 21, 243, 28, 244, 42, 245, 34, 246, 26, 247, 18, 248, 9 
         ], [ 39, 1, 41, 1, 42, 1, 43, 1, 44, 1, 45, 1, 48, 1 ] ], 
  [ [ 151, 8, 152, 9, 156, 5, 166, 8, 170, 5, 181, 1 ], 
      [ 241, 16, 242, 23, 243, 31, 244, 46, 245, 37, 246, 28, 247, 19, 248, 
          10 ], [ 31, 1, 32, 1, 36, 1, 46, 1, 50, 1, 61, 1 ] ], 
  [ [ 142, 10, 147, -2, 158, 2, 159, 2, 162, -5, 164, 3, 168, 7, 169, 2, 179, 
          5 ], 
      [ 241, 15, 242, 22, 243, 30, 244, 44, 245, 36, 246, 28, 247, 19, 248, 
          10 ], [ 22, 1, 38, 1, 39, 1, 44, 2, 48, 1, 49, 1, 59, 1 ] ], 
  [ [ 145, 3, 154, 6, 157, 6, 158, 6, 159, 4, 160, 3, 163, 10 ], 
      [ 241, 16, 242, 24, 243, 32, 244, 47, 245, 38, 246, 29, 247, 20, 248, 
          10 ], [ 25, 1, 34, 1, 37, 1, 38, 1, 39, 1, 40, 1, 43, 1 ] ], 
  [ [ 148, 1, 151, 8, 157, 8, 161, 9, 162, 5, 163, 5, 168, 1 ], 
      [ 241, 16, 242, 23, 243, 31, 244, 46, 245, 38, 246, 29, 247, 20, 248, 
          10 ], [ 28, 1, 31, 1, 37, 1, 41, 1, 42, 1, 43, 1, 48, 1 ] ], 
  [ [ 140, 5, 157, 9, 158, 5, 159, 2, 161, 8, 162, 2, 163, 8 ], 
      [ 241, 16, 242, 24, 243, 32, 244, 48, 245, 39, 246, 30, 247, 20, 248, 
          10 ], [ 20, 1, 37, 1, 38, 1, 39, 1, 41, 1, 42, 1, 43, 1 ] ], 
  [ [ 128, 14, 154, 8, 155, 18, 157, 10, 158, 10, 168, 1 ], 
      [ 241, 20, 242, 29, 243, 39, 244, 58, 245, 48, 246, 37, 247, 26, 248, 
          14 ], [ 8, 1, 34, 1, 35, 1, 37, 1, 38, 1, 48, 1 ] ], 
  [ [ 125, 5, 158, 2, 159, 9, 160, 8, 161, 1, 162, 8, 163, 2, 164, 5 ], 
      [ 241, 16, 242, 24, 243, 32, 244, 48, 245, 40, 246, 30, 247, 20, 248, 
          10 ], [ 5, 1, 38, 1, 39, 1, 40, 1, 41, 1, 42, 1, 43, 1, 44, 1 ] ], 
  [ [ 143, 10, 144, 10, 153, 12, 154, 6, 155, 6, 156, 12 ], 
      [ 241, 20, 242, 28, 243, 38, 244, 56, 245, 46, 246, 36, 247, 24, 248, 
          12 ], [ 23, 1, 24, 1, 33, 1, 34, 1, 35, 1, 36, 1 ] ], 
  [ [ 128, 14, 145, 1, 154, 8, 155, 18, 157, 10, 158, 10, 160, 1 ], 
      [ 241, 20, 242, 30, 243, 40, 244, 59, 245, 48, 246, 37, 247, 26, 248, 
          14 ], [ 8, 1, 25, 1, 34, 1, 35, 1, 37, 1, 38, 1, 40, 1 ] ], 
  [ [ 143, 10, 144, 10, 152, 1, 153, 12, 154, 6, 155, 6, 156, 12 ], 
      [ 241, 20, 242, 29, 243, 39, 244, 58, 245, 47, 246, 36, 247, 24, 248, 
          12 ], [ 23, 1, 24, 1, 32, 1, 33, 1, 34, 1, 35, 1, 36, 1 ] ], 
  [ [ 128, 14, 140, 1, 152, 2, 153, -2, 154, 8, 155, 18, 157, 9, 158, 11, 
          159, -9 ], 
      [ 241, 20, 242, 30, 243, 40, 244, 60, 245, 49, 246, 38, 247, 26, 248, 
          14 ], [ 8, 1, 20, 1, 32, 1, 34, 1, 35, 1, 37, 1, 38, 1 ] ], 
  [ [ 121, 12, 128, 16, 140, 7, 153, 7, 154, 15, 155, 15, 164, 12 ], 
      [ 241, 24, 242, 34, 243, 46, 244, 68, 245, 56, 246, 44, 247, 30, 248, 
          16 ], [ 1, 1, 8, 1, 20, 1, 33, 1, 34, 1, 35, 1, 44, 1 ] ], 
  [ [ 128, 14, 146, 10, 147, 2, 150, 2, 152, 10, 159, 18, 161, 8 ], 
      [ 241, 20, 242, 30, 243, 40, 244, 60, 245, 50, 246, 38, 247, 26, 248, 
          14 ], [ 8, 1, 26, 1, 27, 1, 30, 1, 32, 1, 39, 1, 41, 1 ] ], 
  [ [ 121, 28, 128, 18, 152, 15, 153, 15, 154, 10, 155, 24 ], 
      [ 241, 28, 242, 40, 243, 54, 244, 79, 245, 64, 246, 49, 247, 34, 248, 
          18 ], [ 1, 1, 8, 1, 32, 1, 33, 1, 34, 1, 35, 1 ] ], 
  [ [ 121, 16, 127, 42, 128, 22, 146, 30, 147, 30, 164, 16 ], 
      [ 241, 32, 242, 46, 243, 62, 244, 92, 245, 76, 246, 60, 247, 42, 248, 
          22 ], [ 1, 1, 7, 1, 8, 1, 26, 1, 27, 1, 44, 1 ] ], 
  [ [ 143, 18, 144, 4, 146, 3, 147, 10, 148, 10, 149, 14, 152, 8, 161, 3 ], 
      [ 241, 22, 242, 32, 243, 43, 244, 64, 245, 52, 246, 40, 247, 27, 248, 
          14 ], [ 23, 1, 24, 1, 26, 1, 27, 1, 28, 1, 29, 1, 32, 1, 41, 1 ] ], 
  [ [ 134, 15, 143, 12, 144, 12, 146, 7, 147, 7, 149, 15, 152, 16 ], 
      [ 241, 24, 242, 35, 243, 47, 244, 70, 245, 57, 246, 44, 247, 30, 248, 
          15 ], [ 14, 1, 23, 1, 24, 1, 26, 1, 27, 1, 29, 1, 32, 1 ] ], 
  [ [ 135, 16, 136, 12, 146, 15, 147, 15, 148, 7, 150, 12, 152, 1, 161, 7 ], 
      [ 241, 24, 242, 35, 243, 47, 244, 70, 245, 57, 246, 44, 247, 30, 248, 
          16 ], [ 15, 1, 16, 1, 26, 1, 27, 1, 28, 1, 30, 1, 32, 1, 41, 1 ] ], 
  [ [ 129, 28, 135, 18, 146, 24, 147, 10, 148, 15, 152, 15, 161, 1 ], 
      [ 241, 28, 242, 40, 243, 54, 244, 80, 245, 65, 246, 50, 247, 34, 248, 
          18 ], [ 9, 1, 15, 1, 26, 1, 27, 1, 28, 1, 32, 1, 41, 1 ] ], 
  [ [ 124, 8, 128, 12, 132, -4, 135, 16, 143, 6, 145, 14, 146, 12, 150, 4, 
          151, 18, 154, 4, 155, 10, 159, -4 ], 
      [ 241, 24, 242, 36, 243, 48, 244, 72, 245, 58, 246, 44, 247, 30, 248, 
          16 ], [ 4, 1, 15, 1, 23, 1, 25, 1, 26, 1, 31, 1, 34, 1, 35, 1 ] ], 
  [ [ 138, 10, 139, 24, 140, 15, 141, 15, 142, 18, 143, 28, 153, 1, 169, 1 ], 
      [ 241, 28, 242, 40, 243, 54, 244, 80, 245, 66, 246, 50, 247, 34, 248, 
          18 ], [ 18, 1, 19, 1, 20, 1, 21, 1, 22, 1, 23, 1, 33, 1, 49, 1 ] ], 
  [ [ 127, 42, 128, 22, 143, 16, 144, 16, 146, 30, 147, 30, 152, 1 ], 
      [ 241, 32, 242, 47, 243, 63, 244, 94, 245, 77, 246, 60, 247, 42, 248, 
          22 ], [ 7, 1, 8, 1, 23, 1, 24, 1, 26, 1, 27, 1, 32, 1 ] ], 
  [ [ 127, 28, 128, 22, 134, 42, 137, 17, 138, 28, 143, 3, 144, 32, 146, 3, 
          147, 15, 151, -3 ], 
      [ 241, 32, 242, 48, 243, 64, 244, 95, 245, 78, 246, 60, 247, 42, 248, 
          22 ], [ 8, 1, 14, 1, 17, 1, 18, 1, 24, 1, 26, 1, 27, 1 ] ], 
  [ [ 124, 8, 134, 18, 135, 18, 143, 8, 144, 20, 145, 14, 146, 20, 147, 14 ], 
      [ 241, 28, 242, 42, 243, 56, 244, 84, 245, 68, 246, 52, 247, 36, 248, 
          18 ], [ 4, 1, 14, 1, 15, 1, 23, 1, 24, 1, 25, 1, 26, 1, 27, 1 ] ], 
  [ [ 121, 36, 133, 21, 137, 40, 138, 12, 139, 30, 142, 22, 148, 21 ], 
      [ 241, 36, 242, 52, 243, 70, 244, 103, 245, 84, 246, 64, 247, 43, 248, 
          22 ], [ 1, 1, 13, 1, 17, 1, 18, 1, 19, 1, 22, 1, 28, 1 ] ], 
  [ [ 124, 16, 128, 22, 134, 42, 143, 2, 144, 30, 145, 30, 146, 16, 147, 2 ], 
      [ 241, 32, 242, 48, 243, 64, 244, 96, 245, 78, 246, 60, 247, 42, 248, 
          22 ], [ 4, 1, 8, 1, 14, 1, 23, 1, 24, 1, 25, 1, 26, 1, 27, 1 ] ], 
  [ [ 121, 40, 127, 15, 128, 26, 134, 50, 137, 21, 138, 15, 139, 57, 146, 22, 
          147, -22 ], 
      [ 241, 40, 242, 58, 243, 78, 244, 115, 245, 94, 246, 72, 247, 50, 248, 
          26 ], [ 1, 1, 8, 1, 14, 1, 17, 1, 18, 1, 19, 1, 26, 1 ] ], 
  [ [ 121, 36, 124, 12, 135, 2, 137, 30, 138, 2, 139, 40, 141, 22, 142, 22, 
          146, 20, 147, -20 ], 
      [ 241, 36, 242, 52, 243, 70, 244, 104, 245, 84, 246, 64, 247, 44, 248, 
          22 ], [ 1, 1, 4, 1, 17, 1, 18, 1, 19, 1, 21, 1, 22, 1, 26, 1 ] ], 
  [ [ 124, 21, 127, 15, 128, 26, 129, 40, 134, 50, 137, 1, 138, 57, 139, 15, 
          146, -22, 147, 22 ], 
      [ 241, 40, 242, 58, 243, 78, 244, 116, 245, 94, 246, 72, 247, 50, 248, 
          26 ], [ 4, 1, 8, 1, 9, 1, 14, 1, 17, 1, 18, 1, 19, 1, 27, 1 ] ], 
  [ [ 121, 52, 126, 96, 127, 66, 128, 34, 137, 27, 138, 49, 139, 75 ], 
      [ 241, 52, 242, 76, 243, 102, 244, 151, 245, 124, 246, 96, 247, 66, 
          248, 34 ], [ 1, 1, 6, 1, 7, 1, 8, 1, 17, 1, 18, 1, 19, 1 ] ], 
  [ [ 121, 44, 124, 14, 128, 28, 133, 26, 134, 54, 137, 36, 138, 28, 139, 50 ]
        , [ 241, 44, 242, 64, 243, 86, 244, 128, 245, 104, 246, 80, 247, 54, 
          248, 28 ], [ 1, 1, 4, 1, 8, 1, 13, 1, 14, 1, 17, 1, 18, 1, 19, 1 ] ]
    , 
  [ [ 126, 96, 127, 66, 128, 34, 129, 52, 130, 27, 131, 1, 132, 75, 145, 49 ],
      [ 241, 52, 242, 76, 243, 102, 244, 152, 245, 124, 246, 96, 247, 66, 
          248, 34 ], [ 6, 1, 7, 1, 8, 1, 9, 1, 10, 1, 11, 1, 12, 1, 25, 1 ] ],
  [ [ 121, 60, 122, 66, 123, -66, 125, 108, 127, 22, 128, 38, 130, 22, 131, 
          118, 132, 22, 134, 74, 140, 34 ], 
      [ 241, 60, 242, 88, 243, 118, 244, 174, 245, 142, 246, 108, 247, 74, 
          248, 38 ], [ 1, 1, 2, 1, 5, 1, 8, 1, 10, 1, 11, 1, 14, 1, 20, 1 ] ],
  [ [ 121, 72, 122, 38, 123, -38, 125, 172, 126, 132, 127, 90, 128, 46, 130, 
          68, 131, 142, 132, 68 ], 
      [ 241, 72, 242, 106, 243, 142, 244, 210, 245, 172, 246, 132, 247, 90, 
          248, 46 ], [ 1, 1, 2, 1, 5, 1, 6, 1, 7, 1, 8, 1, 10, 1, 11, 1 ] ], 
  [ [ 121, 92, 122, 136, 123, 182, 124, 270, 125, 220, 126, 168, 127, 114, 
          128, 58 ], 
      [ 241, 92, 242, 136, 243, 182, 244, 270, 245, 220, 246, 168, 247, 114, 
          248, 58 ], [ 1, 1, 2, 1, 3, 1, 4, 1, 5, 1, 6, 1, 7, 1, 8, 1 ] ] ];

      elif type[1] = 'A' then

           list:= SLAfcts.weighted_dyn_diags_An(Length(CartanMatrix(RootSystem(L))));
           orbs:= [ ];
           for i in [1..Length(list[1])] do
               if not IsZero(list[1][i]) then
                  o:= NilpotentOrbit( L, list[1][i] );
                  SetOrbitPartition( o, list[2][i] );
                  Add( orbs, o );
               fi;
           od;
           return orbs;

      elif type[1] = 'B' then

           list:= SLAfcts.weighted_dyn_diags_Bn(Length(CartanMatrix(RootSystem(L))));
           orbs:= [ ];
           for i in [1..Length(list[1])] do
               if not IsZero( list[1][i] ) then
                  o:= NilpotentOrbit( L, list[1][i] );
                  SetOrbitPartition( o, list[2][i] );
                  Add( orbs, o );
               fi;
           od;
           return orbs;

      elif type[1] = 'C' then

           list:= SLAfcts.weighted_dyn_diags_Cn(Length(CartanMatrix(RootSystem(L))));
           orbs:= [ ];
           for i in [1..Length(list[1])] do
               if not IsZero( list[1][i] ) then
                  o:= NilpotentOrbit( L, list[1][i] );
                  SetOrbitPartition( o, list[2][i] );
                  Add( orbs, o );
               fi;
           od;
           return orbs;

      elif type[1] = 'D' then

           list:= SLAfcts.weighted_dyn_diags_Dn(Length(CartanMatrix(RootSystem(L))));
           orbs:= [ ];
           for i in [1..Length(list[1])] do
               if not IsZero( list[1][i] ) then
                  o:= NilpotentOrbit( L, list[1][i] );
                  SetOrbitPartition( o, list[2][i] );
                  Add( orbs, o );
               fi;
           od;
           return orbs;

      fi;

      orbs:= [ ];
      for i in [1..Length( list )] do
          o:= NilpotentOrbit( L, list[i] );
          sl2:= [ ];
          for u in elms[i] do
              a:= Zero(L);
              for j in  [1,3..Length(u)-1] do
                  a := a + u[j+1]*Basis(L)[u[j]];
              od;
              Add( sl2, a );
          od;

          SetSL2Triple( o, sl2 );
          Add( orbs, o );
      od;

      return orbs;

end;

SLAfcts.nilporbs1:= function( L )

    # L simple.

    local tL, F, K, f, o, tK, eK, p, orbs, orb, wd, new, exceptional, eL, list, i;

    tL:= CartanType( CartanMatrix( RootSystem(L) ) );
    if Length(tL.types) > 1 then
       Error("The Lie algebra <L> is not simple.");
    fi;

    F:= LeftActingDomain(L);
    K:= SimpleLieAlgebra( tL.types[1][1], tL.types[1][2], F );
    
    exceptional:= tL.types[1][1] in ["E","F","G"];
    if exceptional then
        f:= IsomorphismOfSemisimpleLieAlgebras( K, L );
    fi;
    o:= SLAfcts.nilporbs( K );

    tK:= CartanType( CartanMatrix( RootSystem(K) ) );

    eK:= tK.enumeration[1];
    eL:= tL.enumeration[1];
    list:= [ ];
    for i in [1..Length(eK)] do list[eK[i]]:= eL[i]; od;
    p:= PermList( list );

    orbs:= [ ];
    for orb in o do
        wd:= Permuted( WeightedDynkinDiagram(orb), p );
        new:= NilpotentOrbit( L, wd );
        if exceptional then
           SetSL2Triple( new, List( SL2Triple( orb ), x -> Image( f, x ) ) );
        else
           SetOrbitPartition( new, OrbitPartition( orb ) );
        fi;
        Add( orbs, new );
    od; 

    return orbs;
end;

InstallMethod( IsSLA,
   "for a Lie algebra",
   true, [ IsLieAlgebra ], 0,
   function( L )
   
      local R, t, c, Ks, i, b, bounds, sm, e, pp;

      R:= RootSystem(L);
      if R = fail then return fail; fi;
      t:= CartanType( CartanMatrix( RootSystem(L) ) );
      if Length( t.types ) = 1 then
         return true;
      else
         c:= CanonicalGenerators(R);
         Ks:= [ ];
         for i in [1..Length(t.enumeration)] do
             Add( Ks, UnderlyingLieAlgebra( SLAfcts.rtsys( L, 
                                        List( c, u -> u{t.enumeration[i]} ) ) ) );
         od;
         b:= Concatenation( List( Ks, U -> BasisVectors( Basis(U) ) ) );
         bounds:= [ [1,Dimension(Ks[1])] ];
         for i in [2..Length(Ks)] do
             sm:= Sum( [1..i-1], j -> Dimension( Ks[j] ) );
             Add( bounds, [sm+1,sm+Dimension(Ks[i])] );
         od;
         e:= Concatenation( t.enumeration );
         pp:= [ ];
         for i in [1..Length(e)] do
             Add( pp, Position(e,i) );
         od;
         SetSLAComponents( L, rec( Ks:= Ks, pp:= pp, bounds:= bounds, 
                      bas:= Basis( L, b ) ) );
         return false;
      fi;

end );


InstallOtherMethod( NilpotentOrbits,
   "for a Lie algebra",
   true, [ IsLieAlgebra ], 10,
   function( L )

    # L semi-simple.

     local Ks, wds, wds0, w1, w2, o, wd, K, p, e, pp, i, orbs;

     if IsSLA(L) then
        return SLAfcts.nilporbs1(L);
     else
        Ks:= SLAComponents( L ).Ks;
        wds:= [ [] ];
        for K in Ks do
            o:= SLAfcts.nilporbs1(K);
            wd:= [ List( [1..Length(CartanMatrix(RootSystem(K)))], i -> 0 ) ];
            Append( wd, List( o, WeightedDynkinDiagram ) );
            wds0:= [ ];
            for w1 in wds do
                for w2 in wd do
                    Add( wds0, Concatenation(w1,w2) );
                od;
            od; 
            wds:= wds0;
        od;
        p:= PositionProperty( wds, x -> ForAll( x, IsZero ) );
        RemoveElmList( wds, p );

        wds0:= [ ];
        pp:= SLAComponents(L).pp;
        for w1 in wds do
            Add( wds0, w1{pp} );
        od;
        wds:= wds0;
             
        orbs:= List( wds, w -> NilpotentOrbit( L, w ) );

     fi;
     return orbs;       

end );


InstallMethod( SignatureTable,
"for Lie algebra", true, [IsLieAlgebra], 0,
function( L )

    local o, R, p, tab, x, w, max, dims, r, u, wt, dc, char, it, i, 
          Ci, h, ev, pos, tp, res, en;
    
    o:= NilpotentOrbits(L);
    R:= RootSystem(L);

    tp:= CartanType( CartanMatrix(R) ).types[1];
    if tp[1] in [ "A", "B", "C", "E", "F", "G" ] then

       p:= PositiveRootsNF(R);
       tab:= [ ];
       for x in o do
           w:= WeightedDynkinDiagram(x);
           max:= p[Length(p)]*w;
           if not IsInt( max ) then # hack to make it work with SqrtField...
              max:= max![1][1][1];
           fi;
           dims:= List([1..max+1], u -> 0 );
           for r in p do
               u:= r*w+1;
               if not IsInt( u ) then
                  u:= u![1][1][1];
               fi;
               dims[u]:= dims[u]+1;
           od;
           dims[1]:= 2*dims[1]+Length(CartanMatrix(R));
           Add( tab, [ dims, w ] );
       od;

       return rec( tipo:= "notD", tab:= tab );

    else

       en:= CartanType( CartanMatrix(R) ).enumeration[1];
       wt:= List( CartanMatrix( R ), x -> 0 );
       wt[en[1]]:= 1;
       dc:= DominantCharacter( L, wt );
       char:= [[],[]];
       for i in [1..Length(dc[1])] do
           it:= WeylOrbitIterator( WeylGroup(R), dc[1][i] );
           while not IsDoneIterator( it ) do
              Add( char[1], NextIterator( it ) );
              Add( char[2], dc[2][i] );
           od;
       od;

       Ci:= FamilyObj(o[1])!.invCM;
       tab:= [ ];
       for x in o do

           h:= Ci*WeightedDynkinDiagram(x);

           dims:= [ ];
           for i in [1..Length(char[1])] do
               ev:= h*char[1][i];

               pos:= PositionProperty( dims, y -> y[1]=ev);
               if pos = fail then
                  Add( dims, [ev, char[2][i]] );
               else
                  dims[pos][2]:= dims[pos][2]+char[2][i];
               fi;
           od;
           Sort( dims, function(a,b) return a[1] < b[1]; end );

           Add( tab, [dims, WeightedDynkinDiagram(x)] );
       od;

       res:= rec( tipo:= "D", char1:= char, tab1:= tab, V1:= 
                HighestWeightModule( L, wt ) );

       wt:= List( CartanMatrix( R ), x -> 0 );
       wt[en[Length(wt)]]:= 1;
       dc:= DominantCharacter( L, wt );
       char:= [[],[]];
       for i in [1..Length(dc[1])] do
           it:= WeylOrbitIterator( WeylGroup(R), dc[1][i] );
           while not IsDoneIterator( it ) do
              Add( char[1], NextIterator( it ) );
              Add( char[2], dc[2][i] );
           od;
       od;

       Ci:= FamilyObj(o[1])!.invCM;
       tab:= [ ];
       for x in o do

           h:= Ci*WeightedDynkinDiagram(x);

           dims:= [ ];
           for i in [1..Length(char[1])] do
               ev:= h*char[1][i];

               pos:= PositionProperty( dims, y -> y[1]=ev);
               if pos = fail then
                  Add( dims, [ev, char[2][i]] );
               else
                  dims[pos][2]:= dims[pos][2]+char[2][i];
               fi;
           od;
           Sort( dims, function(a,b) return a[1] < b[1]; end );

           Add( tab, [dims, WeightedDynkinDiagram(x)] );
       od;

       res.char2:= char; res.tab2:= tab;
       res.V2:= HighestWeightModule( L, wt );
       return res;
    fi;   

end );


SLAfcts.wdd:= function( L, x )

      local sl2, K, H, ch, t, cc, tableM, T, adh, possibles,
            k, dim, rank, poss0, l, c1, c2, fac, f, inds1, inds2, ev, pos, id;

      if not x in L then Error("<x> does not lie in <L>"); fi;

      if IsZero(x) then
         return List( CartanMatrix( RootSystem(L) ), x -> 0 );
      fi;

      T:= SignatureTable(L);
      sl2:= SL2Triple( L, x );
      if T.tipo = "notD" then

         T:= T.tab;
         adh:= TransposedMat( AdjointMatrix( Basis(L), sl2[2] ) );
         possibles:= [ ];

         dim:= Dimension(L);
         rank:= RankMat(adh);
         for k in [1..Length(T)] do
             if T[k][1][1] = dim-rank then
                Add( possibles, T[k] );
             fi;
         od;
   
         k:= 1;
         while Length(possibles) > 1 do 
             rank:= RankMat( adh-k*adh^0 );
             poss0:= [ ];
             for l in [1..Length(possibles)] do
                 if possibles[l][1][k+1] = dim-rank then
                    Add( poss0, possibles[l] );
                 fi;
             od;
             possibles:= poss0;
             k:= k+1;
         od; 

         return possibles[1][2];         

      else

         adh:= MatrixOfAction( Basis( T.V1 ), sl2[2] );
         id:= adh^0;
         c1:= [ ];
         k:= 0; ev:=0;
         while ev < Dimension(T.V1) do
            f:= Dimension(T.V1)-RankMat( adh - k*id );
            if f > 0 then
               Add( c1, [ k, f ] );
               if k=0 then
                  ev:= ev + f;
               else
                  ev:= ev+2*f;
                  Add( c1, [ -k, f ] );
               fi;
            fi;
            k:= k+1;
         od;
         Sort( c1, function(a,b) return a[1]<b[1]; end );
         inds1:= Filtered( [1..Length(T.tab1)], x -> T.tab1[x][1]=c1 );
         if Length(inds1) = 1 then
            return T.tab1[ inds1[1] ][2];
         fi;

         adh:= MatrixOfAction( Basis( T.V2 ), sl2[2] );
         id:= adh^0;
         c2:= [ ];
         k:= 0; ev:=0;
         while ev < Dimension(T.V2) do
            f:= Dimension(T.V2)-RankMat( adh - k*id );
            if f > 0 then
               Add( c2, [ k, f ] );
               if k=0 then
                  ev:= ev + f;
               else
                  ev:= ev+2*f;
                  Add( c2, [ -k, f ] );
               fi;
            fi;
            k:= k+1;
         od;
         Sort( c2, function(a,b) return a[1]<b[1]; end );
         inds2:= Filtered( [1..Length(T.tab2)], x -> T.tab2[x][1]=c2 );

         return T.tab1[ Intersection( inds1, inds2 )[1] ][2];

      fi;
end;

InstallOtherMethod( WeightedDynkinDiagram,
   "for a semisimple Lie algebra and a nilpotent element",
   true, [ IsLieAlgebra, IsObject ], 0,
   function(L,x)

     local r, wd, cf, i, c0, u;

     if IsSLA(L) then
        return SLAfcts.wdd(L,x);
     else
        r:= SLAComponents( L );
        wd:= [ ];
        cf:= Coefficients( r.bas, x );
        for i in [1..Length(r.Ks)] do
            c0:= cf{[r.bounds[i][1]..r.bounds[i][2]]};
            u:= Sum( [1..Length(c0)], k -> c0[k]*Basis(r.Ks[i])[k] );
            Append( wd, SLAfcts.wdd( r.Ks[i], u ) );
        od;
        return wd{r.pp};
     fi;        

end );

InstallMethod( SL2Grading,
"for Lie algebra and element", true, [IsLieAlgebra, IsObject], 0,
function( L, h )

   # Here o  is a nilpotent orbit. We return the grading of the Lie
   # algebra corresponding to "h". Three lists are returned:
   # first the components with grades 1,2,3,... then a list with
   # the components with grades -1,-2,-3... Then the zero component.

   local adh, pos, neg, k, done, sp;

   adh:= AdjointMatrix( Basis(L), h );
   
   pos:= [ ];
   neg:= [ ];
   k:= 1;
   done:= false;
   while not done do
      sp:= NullspaceMat( TransposedMat( adh-k*adh^0 ) );
      Add( pos, List( sp, c -> LinearCombination( Basis(L), c ) ) );
      sp:= NullspaceMat( TransposedMat( adh+k*adh^0 ) );
      Add( neg, List( sp, c -> LinearCombination( Basis(L), c ) ) );

      if Length(pos)>=2 and (pos[ Length(pos) ] = [ ] and pos[ Length(pos)-1 ] = [ ]) then
         done:= true;
      else
         k:= k+1;
      fi;

   od;
   while pos[Length(pos)] = [ ] do Unbind( pos[Length(pos)] ); od;
   while neg[Length(neg)] = [ ] do Unbind( neg[Length(neg)] ); od;

   return [ pos, neg, ShallowCopy( BasisVectors( 
             Basis( LieCentralizer( L, Subalgebra( L,[h] ) ) ) ) ) ];

end );

SLAfcts.rigid_diags_Dn:= function( n )


    local part, good, p, q, cur, k, m, is_even, no_gap, odd_good, u, diags, 
          u0, i, d, L, h, Ci, dims, h0, mat, dim;

    part:= Partitions( 2*n );
    good:= [ ];
    for p in part do
        q:= [ ];
        cur:= p[1];
        k:= 1;
        m:= 0;
        while k <= Length(p) do
            m:= m+1;
            if k < Length(p) and p[k+1] <> cur then
               Add( q, [ cur, m ] );
               cur:= p[k+1];
               m:= 0;
            fi;
            k:= k+1;
        od;
        Add( q, [ cur,m ] );

        is_even:= true;
        for k in [1..Length(q)] do
            if IsEvenInt( q[k][1] ) and not IsEvenInt( q[k][2] ) then
               is_even:= false;
               break;
            fi;
        od;

        no_gap:= true;
        for k in [1..Length(q)-1] do
            if q[k][1] > q[k+1][1]+1 then 
               no_gap:= false;
               break;
            fi;
        od;
        if no_gap then
           no_gap:= q[Length(q)][1] = 1;
        fi;

        odd_good:= true;        
        for k in [1..Length(q)] do
            if IsOddInt(q[k][1]) and q[k][2] = 2 then
               odd_good:= false;
               break;
            fi;
        od;

        if is_even and no_gap and odd_good then Add( good, q ); fi;
    od;
    
    diags:= [ ];
    for q in good do
        u:= [ ];
        for k in [1..Length(q)] do
           u0:= [ ];
           d:= q[k][1]-1;
           while d >= 0 do
              Add( u0, d );
              d:= d-2;
           od;
           for i in [1..q[k][2]] do Append( u, u0 ); od;
        od;
                
        Sort(u);
        u:= Reversed( u );

        d:= [ ];
        for k in [1..n-1] do
            Add( d, u[k]-u[k+1] );
        od;
        Add( d, u[n-1]+u[n] );
        Add( diags, d );
    od;

    L:= SimpleLieAlgebra( "D", n, Rationals );
    h:= ChevalleyBasis(L)[3];
    Ci:= CartanMatrix( RootSystem(L) )^-1;
    dims:= [ ];
    for d in diags do
        h0:= (Ci*d)*h;
        mat:= AdjointMatrix( Basis(L), h0 );
        dim:= Dimension(L);
        dim:= dim - Length( NullspaceMat(mat) );
        dim:= dim - Length( NullspaceMat( mat-IdentityMat( Dimension(L) ) ) );
        Add( dims, dim );
    od;

    return [diags,dims,good];

end;

SLAfcts.rigid_diags_Bn:= function( n )


    local part, good, p, q, cur, k, m, is_even, no_gap, odd_good, u, diags, 
          u0, i, d, L, h, Ci, dims, h0, mat, dim;

    part:= Partitions( 2*n+1 );
    good:= [ ];
    for p in part do
        q:= [ ];
        cur:= p[1];
        k:= 1;
        m:= 0;
        while k <= Length(p) do
            m:= m+1;
            if k < Length(p) and p[k+1] <> cur then
               Add( q, [ cur, m ] );
               cur:= p[k+1];
               m:= 0;
            fi;
            k:= k+1;
        od;
        Add( q, [ cur,m ] );

        is_even:= true;
        for k in [1..Length(q)] do
            if IsEvenInt( q[k][1] ) and not IsEvenInt( q[k][2] ) then
               is_even:= false;
               break;
            fi;
        od;

        no_gap:= true;
        for k in [1..Length(q)-1] do
            if q[k][1] > q[k+1][1]+1 then 
               no_gap:= false;
               break;
            fi;
        od;
        if no_gap then
           no_gap:= q[Length(q)][1] = 1;
        fi;

        odd_good:= true;        
        for k in [1..Length(q)] do
            if IsOddInt(q[k][1]) and q[k][2] = 2 then
               odd_good:= false;
               break;
            fi;
        od;

        if is_even and no_gap and odd_good then Add( good, q ); fi;
    od;
    
    diags:= [ ];
    for q in good do
        u:= [ ];
        for k in [1..Length(q)] do
           u0:= [ ];
           d:= q[k][1]-1;
           while d >= 0 do
              Add( u0, d );
              d:= d-2;
           od;
           for i in [1..q[k][2]] do Append( u, u0 ); od;
        od;

        # get rid of a zero...
        for k in [1..Length(u)] do
            if u[k] = 0 then RemoveElmList( u, k ); break; fi;
        od;
                
        Sort(u);
        u:= Reversed( u );

        d:= [ ];
        for k in [1..n-1] do
            Add( d, u[k]-u[k+1] );
        od;
        Add( d, u[n] );
        Add( diags, d );
    od;

    L:= SimpleLieAlgebra( "B", n, Rationals );
    h:= ChevalleyBasis(L)[3];
    Ci:= CartanMatrix( RootSystem(L) )^-1;
    dims:= [ ];
    for d in diags do
        h0:= (Ci*d)*h;
        mat:= AdjointMatrix( Basis(L), h0 );
        dim:= Dimension(L);
        dim:= dim - Length( NullspaceMat(mat) );
        dim:= dim - Length( NullspaceMat( mat-IdentityMat( Dimension(L) ) ) );
        Add( dims, dim );
    od;

    return [diags,dims,good];

end;

SLAfcts.rigid_diags_Cn:= function( n )


    local part, good, p, q, cur, k, m, is_even, no_gap, odd_good, u, diags, 
          u0, i, d, L, h, Ci, dims, h0, mat, dim;

    part:= Partitions( 2*n );
    good:= [ ];
    for p in part do
        q:= [ ];
        cur:= p[1];
        k:= 1;
        m:= 0;
        while k <= Length(p) do
            m:= m+1;
            if k < Length(p) and p[k+1] <> cur then
               Add( q, [ cur, m ] );
               cur:= p[k+1];
               m:= 0;
            fi;
            k:= k+1;
        od;
        Add( q, [ cur,m ] );

        is_even:= true;
        for k in [1..Length(q)] do
            if IsOddInt( q[k][1] ) and not IsEvenInt( q[k][2] ) then
               is_even:= false;
               break;
            fi;
        od;

        no_gap:= true;
        for k in [1..Length(q)-1] do
            if q[k][1] > q[k+1][1]+1 then 
               no_gap:= false;
               break;
            fi;
        od;
        if no_gap then
           no_gap:= q[Length(q)][1] = 1;
        fi;

        odd_good:= true;        
        for k in [1..Length(q)] do
            if IsEvenInt(q[k][1]) and q[k][2] = 2 then
               odd_good:= false;
               break;
            fi;
        od;

        if is_even and no_gap and odd_good then Add( good, q ); fi;
    od;
    
    diags:= [ ];
    for q in good do
        u:= [ ];
        for k in [1..Length(q)] do
           u0:= [ ];
           d:= q[k][1]-1;
           while d >= 0 do
              Add( u0, d );
              d:= d-2;
           od;
           for i in [1..q[k][2]] do Append( u, u0 ); od;
        od;
                
        Sort(u);
        u:= Reversed( u );

        d:= [ ];
        for k in [1..n-1] do
            Add( d, u[k]-u[k+1] );
        od;
        Add( d, 2*u[n] );
        Add( diags, d );
    od;

    L:= SimpleLieAlgebra( "C", n, Rationals );
    h:= ChevalleyBasis(L)[3];
    Ci:= CartanMatrix( RootSystem(L) )^-1;
    dims:= [ ];
    for d in diags do
        h0:= (Ci*d)*h;
        mat:= AdjointMatrix( Basis(L), h0 );
        dim:= Dimension(L);
        dim:= dim - Length( NullspaceMat(mat) );
        dim:= dim - Length( NullspaceMat( mat-IdentityMat( Dimension(L) ) ) );
        Add( dims, dim );
    od;

    return [diags,dims,good];

end;


SLAfcts.indtableAn:= function( L ) # L of type An....

     local o, rank, p, T, d, sd, i, k, dt, pos;

     o:= NilpotentOrbits(L);
     rank:= Length( CartanMatrix( RootSystem(L) ) );
     p:= [ List( [1..rank+1], x -> 1 ) ];
     Append( p, List( o, OrbitPartition ) );

     T:= [ ];
     for d in p do

         if Length(d) > 1 then
            sd:= List( [1..rank], x -> 0 );
            for i in [1..Length(d)-1] do
                k:= Sum( d{[1..i]} );
                sd[k]:= 2;
            od;

            dt:=[ ];
            for i in [1..rank+1] do
                dt[i]:= Length( Filtered( [1..Length(d)], j -> d[j] >= i ) );
            od;
            for i in [1..Length(dt)] do if dt[i] = 0 then Unbind( dt[i] ); fi; od;
            dt:= Filtered( dt, x -> IsBound(x) );

            pos:= Position( p, dt );
            Add( T, [ sd, WeightedDynkinDiagram(o[pos-1]) ] );
         fi;
     od; 
     
     return T;

end;

InstallMethod( InducedNilpotentOrbits,
"for Lie algebra", true, [IsLieAlgebra], 0,
function( L )

     local tp, T, rank, i, S, pos, o, e, f, wt;

     if not IsSLA(L) then
        Error("currently this only works for simple Lie algebras.");
     fi;

     tp:= CartanType( CartanMatrix( RootSystem(L) ) );
     rank:= tp.types[1][2];

     if tp.types[1][1] = "A" then
        T:= SLAfcts.indtableAn( SimpleLieAlgebra( "A", rank, Rationals ) );
     elif tp.types[1][1] = "B" then
        T:= SLAfcts.SB[rank];
     elif tp.types[1][1] = "C" then
        T:= SLAfcts.SC[rank];
     elif tp.types[1][1] = "D" then
        T:= SLAfcts.SD[rank];        
     elif tp.types[1][1] = "E" then
        T:= SLAfcts.SE[rank];
     elif tp.types[1][1] = "F" then
        T:= SLAfcts.SF[rank];
     elif tp.types[1][1] = "G" then
        T:= SLAfcts.SG[rank];
     fi;

     wt:= List( [1..rank], x -> 2 );
     T:= Concatenation( [[ ShallowCopy(wt), ShallowCopy(wt) ]], T );

     e:= tp.enumeration[1];
     f:= [ ];
     for i in [1..rank] do f[ e[i] ]:= i; od;

     for i in [1..Length(T)] do
         T[i][1]:= T[i][1]{f};
         T[i][2]:= T[i][2]{f};
     od;

     S:= [ ];
     o:= NilpotentOrbits(L);
     for i in [1..Length(T)] do
         pos:= PositionProperty( o, x -> WeightedDynkinDiagram(x) = T[i][2] );
         Add( S, rec( sheetdiag:= T[i][1], norbit:= o[pos] ) );
     od;

     return S;

end );

InstallMethod( RigidNilpotentOrbits,
"for Lie algebra", true, [IsLieAlgebra], 0,
function( L )

     local tp, T, rank, i, S, pos, o, e, f;

     if not IsSLA(L) then
        Error("currently this only works for simple Lie algebras.");
     fi;

     tp:= CartanType( CartanMatrix( RootSystem(L) ) );
     if tp.types[1][1] = "A" then
        T:= [ ];
     else
        rank:= tp.types[1][2];
        if tp.types[1][1] = "B" then
           T:= SLAfcts.rigid_diags_Bn( rank )[1];
        elif tp.types[1][1] = "C" then
           T:= SLAfcts.rigid_diags_Cn( rank )[1];
        elif tp.types[1][1] = "D" then
           T:= SLAfcts.rigid_diags_Dn( rank )[1];        
        elif tp.types[1][1] = "E" then
           if rank = 6 then
              T:= [ [1,0,0,1,0,1], [0,0,0,1,0,0], [0,1,0,0,0,0] ];
           elif rank = 7 then
              T:= [ [1,0,0,1,0,0,0], [0,0,1,0,0,1,0], 
                    [0,0,0,1,0,0,0], [0,1,0,0,0,0,1],
                    [0,0,1,0,0,0,0], [0,0,0,0,0,1,0],
                    [0,1,0,0,0,0,0] ];
           elif rank = 8 then
              T:= [ [ 0, 0, 0, 0, 0, 0, 0, 1 ], [ 1, 0, 0, 0, 0, 0, 0, 0 ], 
                    [ 0, 0, 0, 0, 0, 0, 1, 0 ], [ 0, 1, 0, 0, 0, 0, 0, 0 ], 
                    [ 1, 0, 0, 0, 0, 0, 0, 1 ], [ 0, 0, 0, 0, 0, 1, 0, 0 ], 
                    [ 0, 0, 1, 0, 0, 0, 0, 0 ], [ 1, 0, 0, 0, 0, 0, 1, 0 ], 
                    [ 0, 0, 0, 0, 0, 1, 0, 1 ], [ 0, 0, 0, 0, 1, 0, 0, 0 ], 
                    [ 0, 0, 1, 0, 0, 0, 0, 1 ], [ 0, 1, 0, 0, 0, 0, 1, 0 ], 
                    [ 0, 0, 0, 1, 0, 0, 0, 0 ], [ 1, 0, 0, 0, 1, 0, 0, 0 ], 
                    [ 0, 0, 0, 1, 0, 0, 1, 0 ], [ 1, 0, 0, 1, 0, 0, 0, 1 ], 
                    [ 0, 0, 1, 0, 0, 1, 0, 1 ] ];
           fi;
        elif tp.types[1][1] = "F" then
           T:= [ [ 0, 1, 0, 0 ], [ 1, 0, 0, 0 ], [ 0, 0, 0, 1 ], [ 0, 0, 1, 0 ], [ 1, 0, 0, 1 ] ];
        elif tp.types[1][1] = "G" then
           T:= [ [ 1, 0 ], [ 0, 1 ] ];
        fi;

        e:= tp.enumeration[1];
        f:= [ ];
        for i in [1..rank] do f[ e[i] ]:= i; od;

        for i in [1..Length(T)] do
            T[i]:= T[i]{f};
        od;
     fi;

     o:= NilpotentOrbits(L);
     return Filtered( o, x -> WeightedDynkinDiagram(x) in T );

end );

