"use strict";
/*
  Simplified BSD License (see LICENSES/BSD-2-Clause.txt or https://opensource.org/licenses/BSD-2-Clause)
  SPDX-FileCopyrightText: Ansible Project
  SPDX-License-Identifier: BSD-2-Clause
*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.toMD = exports.quoteMD = void 0;
// CommonMark spec: https://spec.commonmark.org/current/
const opts_1 = require("./opts");
const html_1 = require("./html");
const format_1 = require("./format");
function quoteMD(text) {
    return text.replace(/([!"#$%&'()*+,:;<=>?@[\\\]^_`{|}~.-])/g, '\\$1');
}
exports.quoteMD = quoteMD;
function formatOptionLike(part, url, what) {
    let link_start = '';
    let link_end = '';
    if (url) {
        link_start = `<a href="${(0, html_1.quoteHTMLArg)(encodeURI(url))}">`;
        link_end = '</a>';
    }
    let strong_start = '';
    let strong_end = '';
    if (what === 'option') {
        if (part.value === undefined) {
            strong_start = '<strong>';
            strong_end = '</strong>';
        }
    }
    let text;
    if (part.value === undefined) {
        text = part.name;
    }
    else {
        text = `${part.name}=${part.value}`;
    }
    return `<code>${strong_start}${link_start}${quoteMD(text)}${link_end}${strong_end}</code>`;
}
const DEFAULT_FORMATTER = {
    formatError: (part) => `<b>ERROR while parsing</b>: ${quoteMD(part.message)}`,
    formatBold: (part) => `<b>${quoteMD(part.text)}</b>`,
    formatCode: (part) => `<code>${quoteMD(part.text)}</code>`,
    formatHorizontalLine: () => '<hr>',
    formatItalic: (part) => `<em>${quoteMD(part.text)}</em>`,
    formatLink: (part) => `[${quoteMD(part.text)}](${encodeURI(part.url)})`,
    formatModule: (part, url) => (url ? `[${quoteMD(part.fqcn)}](${quoteMD(encodeURI(url))})` : `${quoteMD(part.fqcn)}`),
    formatRSTRef: (part) => `${quoteMD(part.text)}`,
    formatURL: (part) => `[${quoteMD(encodeURI(part.url))}](${encodeURI(part.url)})`,
    formatText: (part) => quoteMD(part.text),
    formatEnvVariable: (part) => `<code>${quoteMD(part.name)}</code>`,
    formatOptionName: (part, url) => formatOptionLike(part, url, 'option'),
    formatOptionValue: (part) => `<code>${quoteMD(part.value)}</code>`,
    formatPlugin: (part, url) => url ? `[${quoteMD(part.plugin.fqcn)}](${quoteMD(encodeURI(url))})` : `${quoteMD(part.plugin.fqcn)}`,
    formatReturnValue: (part, url) => formatOptionLike(part, url, 'retval'),
};
function toMD(paragraphs, opts) {
    const mergedOpts = (0, opts_1.mergeOpts)(opts !== null && opts !== void 0 ? opts : {}, DEFAULT_FORMATTER);
    const result = [];
    for (const paragraph of paragraphs) {
        const line = [];
        (0, format_1.addToDestination)(line, paragraph, mergedOpts);
        if (!line.length) {
            line.push(' ');
        }
        result.push(line.join(''));
    }
    return result.join('\n\n');
}
exports.toMD = toMD;
//# sourceMappingURL=md.js.map