﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codepipeline/CodePipeline_EXPORTS.h>
#include <aws/codepipeline/CodePipelineRequest.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/codepipeline/model/CurrentRevision.h>
#include <aws/codepipeline/model/ExecutionDetails.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <utility>

namespace Aws
{
namespace CodePipeline
{
namespace Model
{

  /**
   * <p>Represents the input of a <code>PutJobSuccessResult</code>
   * action.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/PutJobSuccessResultInput">AWS
   * API Reference</a></p>
   */
  class PutJobSuccessResultRequest : public CodePipelineRequest
  {
  public:
    AWS_CODEPIPELINE_API PutJobSuccessResultRequest() = default;

    // Service request name is the Operation name which will send this request out,
    // each operation should has unique request name, so that we can get operation's name from this request.
    // Note: this is not true for response, multiple operations may have the same response name,
    // so we can not get operation's name from response.
    inline virtual const char* GetServiceRequestName() const override { return "PutJobSuccessResult"; }

    AWS_CODEPIPELINE_API Aws::String SerializePayload() const override;

    AWS_CODEPIPELINE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;


    ///@{
    /**
     * <p>The unique system-generated ID of the job that succeeded. This is the same ID
     * returned from <code>PollForJobs</code>.</p>
     */
    inline const Aws::String& GetJobId() const { return m_jobId; }
    inline bool JobIdHasBeenSet() const { return m_jobIdHasBeenSet; }
    template<typename JobIdT = Aws::String>
    void SetJobId(JobIdT&& value) { m_jobIdHasBeenSet = true; m_jobId = std::forward<JobIdT>(value); }
    template<typename JobIdT = Aws::String>
    PutJobSuccessResultRequest& WithJobId(JobIdT&& value) { SetJobId(std::forward<JobIdT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>The ID of the current revision of the artifact successfully worked on by the
     * job.</p>
     */
    inline const CurrentRevision& GetCurrentRevision() const { return m_currentRevision; }
    inline bool CurrentRevisionHasBeenSet() const { return m_currentRevisionHasBeenSet; }
    template<typename CurrentRevisionT = CurrentRevision>
    void SetCurrentRevision(CurrentRevisionT&& value) { m_currentRevisionHasBeenSet = true; m_currentRevision = std::forward<CurrentRevisionT>(value); }
    template<typename CurrentRevisionT = CurrentRevision>
    PutJobSuccessResultRequest& WithCurrentRevision(CurrentRevisionT&& value) { SetCurrentRevision(std::forward<CurrentRevisionT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>A token generated by a job worker, such as a CodeDeploy deployment ID, that a
     * successful job provides to identify a custom action in progress. Future jobs use
     * this token to identify the running instance of the action. It can be reused to
     * return more information about the progress of the custom action. When the action
     * is complete, no continuation token should be supplied.</p>
     */
    inline const Aws::String& GetContinuationToken() const { return m_continuationToken; }
    inline bool ContinuationTokenHasBeenSet() const { return m_continuationTokenHasBeenSet; }
    template<typename ContinuationTokenT = Aws::String>
    void SetContinuationToken(ContinuationTokenT&& value) { m_continuationTokenHasBeenSet = true; m_continuationToken = std::forward<ContinuationTokenT>(value); }
    template<typename ContinuationTokenT = Aws::String>
    PutJobSuccessResultRequest& WithContinuationToken(ContinuationTokenT&& value) { SetContinuationToken(std::forward<ContinuationTokenT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>The execution details of the successful job, such as the actions taken by the
     * job worker.</p>
     */
    inline const ExecutionDetails& GetExecutionDetails() const { return m_executionDetails; }
    inline bool ExecutionDetailsHasBeenSet() const { return m_executionDetailsHasBeenSet; }
    template<typename ExecutionDetailsT = ExecutionDetails>
    void SetExecutionDetails(ExecutionDetailsT&& value) { m_executionDetailsHasBeenSet = true; m_executionDetails = std::forward<ExecutionDetailsT>(value); }
    template<typename ExecutionDetailsT = ExecutionDetails>
    PutJobSuccessResultRequest& WithExecutionDetails(ExecutionDetailsT&& value) { SetExecutionDetails(std::forward<ExecutionDetailsT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>Key-value pairs produced as output by a job worker that can be made available
     * to a downstream action configuration. <code>outputVariables</code> can be
     * included only when there is no continuation token on the request.</p>
     */
    inline const Aws::Map<Aws::String, Aws::String>& GetOutputVariables() const { return m_outputVariables; }
    inline bool OutputVariablesHasBeenSet() const { return m_outputVariablesHasBeenSet; }
    template<typename OutputVariablesT = Aws::Map<Aws::String, Aws::String>>
    void SetOutputVariables(OutputVariablesT&& value) { m_outputVariablesHasBeenSet = true; m_outputVariables = std::forward<OutputVariablesT>(value); }
    template<typename OutputVariablesT = Aws::Map<Aws::String, Aws::String>>
    PutJobSuccessResultRequest& WithOutputVariables(OutputVariablesT&& value) { SetOutputVariables(std::forward<OutputVariablesT>(value)); return *this;}
    template<typename OutputVariablesKeyT = Aws::String, typename OutputVariablesValueT = Aws::String>
    PutJobSuccessResultRequest& AddOutputVariables(OutputVariablesKeyT&& key, OutputVariablesValueT&& value) {
      m_outputVariablesHasBeenSet = true; m_outputVariables.emplace(std::forward<OutputVariablesKeyT>(key), std::forward<OutputVariablesValueT>(value)); return *this;
    }
    ///@}
  private:

    Aws::String m_jobId;
    bool m_jobIdHasBeenSet = false;

    CurrentRevision m_currentRevision;
    bool m_currentRevisionHasBeenSet = false;

    Aws::String m_continuationToken;
    bool m_continuationTokenHasBeenSet = false;

    ExecutionDetails m_executionDetails;
    bool m_executionDetailsHasBeenSet = false;

    Aws::Map<Aws::String, Aws::String> m_outputVariables;
    bool m_outputVariablesHasBeenSet = false;
  };

} // namespace Model
} // namespace CodePipeline
} // namespace Aws
