/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.util.compress;

import java.io.IOException;
import java.util.Arrays;
import java.util.Objects;
import org.apache.lucene.store.DataInput;
import org.apache.lucene.store.DataOutput;
import org.apache.lucene.util.BitUtil;

public final class LZ4 {
    public static final int MAX_DISTANCE = 65536;
    static final int MEMORY_USAGE = 14;
    static final int MIN_MATCH = 4;
    static final int LAST_LITERALS = 5;
    static final int HASH_LOG_HC = 15;
    static final int HASH_TABLE_SIZE_HC = 32768;

    private LZ4() {
    }

    private static int hash(int i, int hashBits) {
        return i * -1640531535 >>> 32 - hashBits;
    }

    private static int hashHC(int i) {
        return LZ4.hash(i, 15);
    }

    private static int readInt(byte[] buf, int i) {
        return BitUtil.VH_LE_INT.get(buf, i);
    }

    private static int commonBytes(byte[] b, int o1, int o2, int limit) {
        assert (o1 < o2);
        return Arrays.mismatch(b, o1, limit, b, o2, limit);
    }

    public static int decompress(DataInput compressed, int decompressedLen, byte[] dest, int dOff) throws IOException {
        int destEnd = dOff + decompressedLen;
        do {
            int token;
            int literalLen;
            if ((literalLen = (token = compressed.readByte() & 0xFF) >>> 4) != 0) {
                if (literalLen == 15) {
                    byte len;
                    while ((len = compressed.readByte()) == -1) {
                        literalLen += 255;
                    }
                    literalLen += len & 0xFF;
                }
                compressed.readBytes(dest, dOff, literalLen);
                dOff += literalLen;
            }
            if (dOff >= destEnd) break;
            int matchDec = compressed.readShort() & 0xFFFF;
            assert (matchDec > 0);
            int matchLen = token & 0xF;
            if (matchLen == 15) {
                byte len;
                while ((len = compressed.readByte()) == -1) {
                    matchLen += 255;
                }
                matchLen += len & 0xFF;
            }
            int fastLen = (matchLen += 4) + 7 & 0xFFFFFFF8;
            if (matchDec < matchLen || dOff + fastLen > destEnd) {
                int ref = dOff - matchDec;
                int end = dOff + matchLen;
                while (dOff < end) {
                    dest[dOff] = dest[ref];
                    ++ref;
                    ++dOff;
                }
            } else {
                System.arraycopy(dest, dOff - matchDec, dest, dOff, fastLen);
                dOff += matchLen;
            }
        } while (dOff < destEnd);
        return dOff;
    }

    private static void encodeLen(int l, DataOutput out) throws IOException {
        while (l >= 255) {
            out.writeByte((byte)-1);
            l -= 255;
        }
        out.writeByte((byte)l);
    }

    private static void encodeLiterals(byte[] bytes, int token, int anchor, int literalLen, DataOutput out) throws IOException {
        out.writeByte((byte)token);
        if (literalLen >= 15) {
            LZ4.encodeLen(literalLen - 15, out);
        }
        out.writeBytes(bytes, anchor, literalLen);
    }

    private static void encodeLastLiterals(byte[] bytes, int anchor, int literalLen, DataOutput out) throws IOException {
        int token = Math.min(literalLen, 15) << 4;
        LZ4.encodeLiterals(bytes, token, anchor, literalLen, out);
    }

    private static void encodeSequence(byte[] bytes, int anchor, int matchRef, int matchOff, int matchLen, DataOutput out) throws IOException {
        int literalLen = matchOff - anchor;
        assert (matchLen >= 4);
        int token = Math.min(literalLen, 15) << 4 | Math.min(matchLen - 4, 15);
        LZ4.encodeLiterals(bytes, token, anchor, literalLen, out);
        int matchDec = matchOff - matchRef;
        assert (matchDec > 0 && matchDec < 65536);
        out.writeShort((short)matchDec);
        if (matchLen >= 19) {
            LZ4.encodeLen(matchLen - 15 - 4, out);
        }
    }

    public static void compress(byte[] bytes, int off, int len, DataOutput out, HashTable ht) throws IOException {
        LZ4.compressWithDictionary(bytes, off, 0, len, out, ht);
    }

    public static void compressWithDictionary(byte[] bytes, int dictOff, int dictLen, int len, DataOutput out, HashTable ht) throws IOException {
        int off;
        Objects.checkFromIndexSize(dictOff, dictLen, bytes.length);
        Objects.checkFromIndexSize(dictOff + dictLen, len, bytes.length);
        if (dictLen > 65536) {
            throw new IllegalArgumentException("dictLen must not be greater than 64kB, but got " + dictLen);
        }
        int end = dictOff + dictLen + len;
        int anchor = off = dictOff + dictLen;
        if (len > 9) {
            int limit = end - 5;
            int matchLimit = limit - 4;
            ht.reset(bytes, dictOff, dictLen + len);
            ht.initDictionary(dictLen);
            block0: while (off <= limit) {
                while (off < matchLimit) {
                    int ref = ht.get(off);
                    if (ref != -1) {
                        assert (ref >= dictOff && ref < off);
                        assert (LZ4.readInt(bytes, ref) == LZ4.readInt(bytes, off));
                    } else {
                        ++off;
                        continue;
                    }
                    int matchLen = 4 + LZ4.commonBytes(bytes, ref + 4, off + 4, limit);
                    int r = ht.previous(ref);
                    int min2 = Math.max(off - 65536 + 1, dictOff);
                    while (r >= min2) {
                        assert (LZ4.readInt(bytes, r) == LZ4.readInt(bytes, off));
                        int rMatchLen = 4 + LZ4.commonBytes(bytes, r + 4, off + 4, limit);
                        if (rMatchLen > matchLen) {
                            ref = r;
                            matchLen = rMatchLen;
                        }
                        r = ht.previous(r);
                    }
                    LZ4.encodeSequence(bytes, anchor, ref, off, matchLen, out);
                    anchor = off += matchLen;
                    continue block0;
                }
                break block0;
            }
        }
        int literalLen = end - anchor;
        assert (literalLen >= 5 || literalLen == len);
        LZ4.encodeLastLiterals(bytes, anchor, end - anchor, out);
    }

    public static final class HighCompressionHashTable
    extends HashTable {
        private static final int MAX_ATTEMPTS = 256;
        private static final int MASK = 65535;
        private byte[] bytes;
        private int base;
        private int next;
        private int end;
        private final int[] hashTable = new int[32768];
        private final short[] chainTable;
        private int attempts = 0;

        public HighCompressionHashTable() {
            Arrays.fill(this.hashTable, -1);
            this.chainTable = new short[65536];
            Arrays.fill(this.chainTable, (short)-1);
        }

        @Override
        void reset(byte[] bytes, int off, int len) {
            Objects.checkFromIndexSize(off, len, bytes.length);
            if (this.end - this.base < this.chainTable.length) {
                int endOffset;
                int startOffset = this.base & 0xFFFF;
                int n = endOffset = this.end == 0 ? 0 : (this.end - 1 & 0xFFFF) + 1;
                if (startOffset < endOffset) {
                    Arrays.fill(this.chainTable, startOffset, endOffset, (short)-1);
                } else {
                    Arrays.fill(this.chainTable, 0, endOffset, (short)-1);
                    Arrays.fill(this.chainTable, startOffset, this.chainTable.length, (short)-1);
                }
            } else {
                Arrays.fill(this.hashTable, -1);
                Arrays.fill(this.chainTable, (short)-1);
            }
            this.bytes = bytes;
            this.base = off;
            this.next = off;
            this.end = off + len;
        }

        @Override
        void initDictionary(int dictLen) {
            assert (this.next == this.base);
            for (int i = 0; i < dictLen; ++i) {
                this.addHash(this.base + i);
            }
            this.next += dictLen;
        }

        @Override
        int get(int off) {
            assert (off >= this.next);
            assert (off < this.end);
            while (this.next < off) {
                this.addHash(this.next);
                ++this.next;
            }
            int v = LZ4.readInt(this.bytes, off);
            int h2 = LZ4.hashHC(v);
            this.attempts = 0;
            int ref = this.hashTable[h2];
            if (ref >= off) {
                return -1;
            }
            int min2 = Math.max(this.base, off - 65536 + 1);
            while (ref >= min2 && this.attempts < 256) {
                if (LZ4.readInt(this.bytes, ref) == v) {
                    return ref;
                }
                ref -= this.chainTable[ref & 0xFFFF] & 0xFFFF;
                ++this.attempts;
            }
            return -1;
        }

        private void addHash(int off) {
            int v = LZ4.readInt(this.bytes, off);
            int h2 = LZ4.hashHC(v);
            int delta = off - this.hashTable[h2];
            if (delta <= 0 || delta >= 65536) {
                delta = 65535;
            }
            this.chainTable[off & 0xFFFF] = (short)delta;
            this.hashTable[h2] = off;
        }

        @Override
        int previous(int off) {
            int v = LZ4.readInt(this.bytes, off);
            int ref = off - (this.chainTable[off & 0xFFFF] & 0xFFFF);
            while (ref >= this.base && this.attempts < 256) {
                if (LZ4.readInt(this.bytes, ref) == v) {
                    return ref;
                }
                ref -= this.chainTable[ref & 0xFFFF] & 0xFFFF;
                ++this.attempts;
            }
            return -1;
        }

        @Override
        boolean assertReset() {
            for (int i = 0; i < this.chainTable.length; ++i) {
                assert (this.chainTable[i] == -1) : i;
            }
            return true;
        }
    }

    public static final class FastCompressionHashTable
    extends HashTable {
        private byte[] bytes;
        private int base;
        private int lastOff;
        private int end;
        private int hashLog;
        private Table hashTable;

        @Override
        void reset(byte[] bytes, int off, int len) {
            Objects.checkFromIndexSize(off, len, bytes.length);
            this.bytes = bytes;
            this.base = off;
            this.end = off + len;
            int bitsPerOffset = len - 5 < 65536 ? 16 : 32;
            int bitsPerOffsetLog = 32 - Integer.numberOfLeadingZeros(bitsPerOffset - 1);
            this.hashLog = 17 - bitsPerOffsetLog;
            if (this.hashTable == null || this.hashTable.size() < 1 << this.hashLog || this.hashTable.getBitsPerValue() < bitsPerOffset) {
                if (bitsPerOffset > 16) {
                    assert (bitsPerOffset == 32);
                    this.hashTable = new Table32(1 << this.hashLog);
                } else {
                    assert (bitsPerOffset == 16);
                    this.hashTable = new Table16(1 << this.hashLog);
                }
            }
            this.lastOff = off - 1;
        }

        @Override
        void initDictionary(int dictLen) {
            for (int i = 0; i < dictLen; ++i) {
                int v = LZ4.readInt(this.bytes, this.base + i);
                int h2 = LZ4.hash(v, this.hashLog);
                this.hashTable.set(h2, i);
            }
            this.lastOff += dictLen;
        }

        @Override
        int get(int off) {
            assert (off > this.lastOff);
            assert (off < this.end);
            int v = LZ4.readInt(this.bytes, off);
            int h2 = LZ4.hash(v, this.hashLog);
            int ref = this.base + this.hashTable.getAndSet(h2, off - this.base);
            this.lastOff = off;
            if (ref < off && off - ref < 65536 && LZ4.readInt(this.bytes, ref) == v) {
                return ref;
            }
            return -1;
        }

        @Override
        public int previous(int off) {
            return -1;
        }

        @Override
        boolean assertReset() {
            return true;
        }
    }

    private static class Table32
    extends Table {
        private final int[] table;

        Table32(int size) {
            this.table = new int[size];
        }

        @Override
        void set(int index, int value) {
            this.table[index] = value;
        }

        @Override
        int getAndSet(int index, int value) {
            int prev = this.table[index];
            this.set(index, value);
            return prev;
        }

        @Override
        int getBitsPerValue() {
            return 32;
        }

        @Override
        int size() {
            return this.table.length;
        }
    }

    private static class Table16
    extends Table {
        private final short[] table;

        Table16(int size) {
            this.table = new short[size];
        }

        @Override
        void set(int index, int value) {
            assert (value >= 0 && value < 65536);
            this.table[index] = (short)value;
        }

        @Override
        int getAndSet(int index, int value) {
            int prev = Short.toUnsignedInt(this.table[index]);
            this.set(index, value);
            return prev;
        }

        @Override
        int getBitsPerValue() {
            return 16;
        }

        @Override
        int size() {
            return this.table.length;
        }
    }

    private static abstract class Table {
        private Table() {
        }

        abstract void set(int var1, int var2);

        abstract int getAndSet(int var1, int var2);

        abstract int getBitsPerValue();

        abstract int size();
    }

    static abstract class HashTable {
        HashTable() {
        }

        abstract void reset(byte[] var1, int var2, int var3);

        abstract void initDictionary(int var1);

        abstract int get(int var1);

        abstract int previous(int var1);

        abstract boolean assertReset();
    }
}

