# Configuration file handler
#
# Copyright 2021 cunidev
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from subprocess import call, Popen, getstatusoutput
from os.path import isfile
from shutil import copyfile
from gestures.gesture import Gesture
from re import compile


class ConfigFileHandler:
    def __init__(self, path, version):
        self.userPath = path
        self.appVersion = version
        self.filePath = path + "/.config/libinput-gestures.conf"
        self.backupPath = path + "/.config/libinput-gestures.conf.bak"

    def generate_config_file(self):
        if(isfile(self.filePath)):
            return False
        else:
            try:
                f = open(self.filePath, "w+")
                f.write("# Generated by Gestures ")  # will be overwritten later, just temporary header
                f.close()
                return True
            except Exception as e:
                print(e)
                return False

    def openFile(self):
        self.swipe_threshold = 0

        self.validSupportedRegex = compile(
            "^\s*(#D:\s)?gesture\s((swipe)\s(up|down|left|right|left_up|left_down|right_up|right_down)\s([3-4]\s)?|(pinch\s(in|out|clockwise|anticlockwise)\s([2-4]\s)?))\s*(.+)")
        self.validUnsupportedRegex = compile("^\s*device\s")
        self.validSwipeThresholdRegex = compile("^\s*swipe_threshold\s*(\d+)\s*$")
        try:
            self.file = open(self.filePath, "r+")
        except:
            raise Exception("err_file_open")

        self.rawLines = list(self.file)

        for n in range(len(self.rawLines)):
            if(self.rawLines[n][-1:] == '\n'):
                self.rawLines[n] = self.rawLines[n][:-1]

        self.fileLines = ["# Generated by Gestures " + self.appVersion + "  -->  https://gitlab.com/cunidev/gestures",
                          "# Manual editing might result in data loss!"]
        self.validUnsupportedLines = ["# Unsupported lines"]
        self.invalidLines = ["# Invalid lines"]
        self.gestures = []

        for line in self.rawLines:
            validLine = self.validSupportedRegex.match(line)
            if(validLine):
                line = line.replace('\t', ' ')
                line = line.split(' ')
                filter(None, line)

                enabled = True
                if(line[0] == "#D:"):
                    enabled = False
                    line.pop(0)

                type = line[1]
                direction = line[2]

                if(line[3].isnumeric()):
                    fingers = line[3]
                    command = ' '.join(line[4:])

                else:
                    fingers = 4
                    command = ' '.join(line[3:])

                curGesture = Gesture(
                    type, direction, command, fingers, enabled)
                self.gestures.append(curGesture)

            elif(self.validUnsupportedRegex.match(line)):
                self.validUnsupportedLines.append(line)

            elif(self.validSwipeThresholdRegex.match(line)):
                try:
                    swipe_threshold = int(self.validSwipeThresholdRegex.match(line)[1])
                    self.swipe_threshold = swipe_threshold
                except:
                    pass

            elif((line[:1] == "#") and not ("#I: " in line)) or (line == ""):
                pass
            else:
                if ("#I: " in line):
                    self.invalidLines.append(line)
                else:
                    self.invalidLines.append("#I: " + line)

    def reload_file(self):
        self.file.close()
        self.openFile()

    def reload_process(self):
        try:
            Popen(["libinput-gestures-setup", "restart"], shell=False)
        except:
            # maybe we are in a Flatpak sandbox
            # BUG: this never throws error
            try:
                (code, out) = getstatusoutput("flatpak-spawn --host libinput-gestures-setup status")
                if code == 0:
                    Popen(["flatpak-spawn","--host"] + ["libinput-gestures-setup", "restart"], shell=False)
                else:
                    raise Exception("err_exec")
            except:
                raise Exception("err_exec")

    def save(self):
        self.fileLines.append('\n')
        self.fileLines.extend(self.invalidLines)
        self.fileLines.append('\n')
        self.fileLines.extend(self.validUnsupportedLines)

        if(self.swipe_threshold >= 0 and self.swipe_threshold <= 100):
            self.fileLines.append('\n# Swipe threshold (0-100)')
            self.fileLines.append("swipe_threshold " + str(self.swipe_threshold))

        self.fileLines.append('\n# Gestures')

        for curGesture in self.gestures:
            self.fileLines.append(curGesture.make())

        self.file.seek(0)
        self.file.write('\n'.join(self.fileLines))
        self.file.truncate()
        self.reload_file()

    def backup(self):
        return self.export_file(self.backupPath)

    def restore(self):
        try:
            copyfile(self.backupPath, self.filePath)
            return True
        except Exception as e:
            print(e)
            return False

    def import_file(self, path):
        try:
            copyfile(path, self.filePath)
            return True
        except Exception as e:
            print(e)
            return False

    def export_file(self, path):
        try:
            copyfile(self.filePath, path)
            return True
        except Exception as e:
            print(e)
            return False

    def is_valid(self):
        return (
            len(self.rawLines) > 0 and
            "# Generated by Gestures " in self.rawLines[0]
        )

